/*
 * fhist - file history and comparison tools
 * Copyright (C) 2000, 2002, 2008, 2010, 2012 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <common/ac/errno.h>
#include <common/ac/stdio.h>
#include <common/ac/stdlib.h>
#include <libexplain/fclose.h>
#include <libexplain/fflush.h>
#include <libexplain/fopen.h>
#include <libexplain/fwrite.h>
#include <libexplain/putc.h>
#include <libexplain/strdup.h>

#include <common/fcheck.h>
#include <common/output/file.h>
#include <common/output/private.h>
#include <common/output/stdout.h>
#include <common/error_intl.h>


typedef struct output_file_ty output_file_ty;
struct output_file_ty
{
    output_ty       inherited;
    char            *filename;
    FILE            *deeper;
};


static void
destructor(output_ty *fp)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    explain_fclose_or_die(this->deeper);
    this->deeper = 0;
    free(this->filename);
    this->filename = 0;
}


static const char *
filename(output_ty *fp)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    return this->filename;
}


static long
otell(output_ty *fp)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    return ftell(this->deeper);
}


static void
oputc(output_ty *fp, int c)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    explain_putc_or_die(c, this->deeper);
}


static void
owrite(output_ty *fp, const void *data, size_t len)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    explain_fwrite_or_die(data, 1, len, this->deeper);
}


static void
oflush(output_ty *fp)
{
    output_file_ty  *this;

    this = (output_file_ty *)fp;
    explain_fflush_or_die(this->deeper);
}


static output_vtbl_ty vtbl =
{
    sizeof(output_file_ty),
    destructor,
    filename,
    otell,
    oputc,
    output_generic_fputs,
    owrite,
    oflush,
    "file",
};


static output_ty *
output_file_open(const char *fn, const char *mode)
{
    output_ty       *result;
    output_file_ty  *this;

    if (!fn || !*fn)
        return output_stdout();
    result = output_new(&vtbl);
    this = (output_file_ty *)result;
    this->deeper = explain_fopen_or_die(fn, mode);
    this->filename = explain_strdup_or_die(fn);
    return result;
}


output_ty *
output_file_text_open(const char *fn)
{
    return output_file_open(fn, "w");
}


output_ty *
output_file_binary_open(const char *fn)
{
    return output_file_open(fn, "wb");
}


/* vim: set ts=8 sw=4 et : */
