/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 */

/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

/*
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

package org.opensearch.common;

import org.opensearch.test.OpenSearchTestCase;

import java.nio.charset.StandardCharsets;

import static org.hamcrest.Matchers.is;

public class CharArraysTests extends OpenSearchTestCase {

    public void testCharsToBytes() {
        final String originalValue = randomUnicodeOfCodepointLengthBetween(0, 32);
        final byte[] expectedBytes = originalValue.getBytes(StandardCharsets.UTF_8);
        final char[] valueChars = originalValue.toCharArray();

        final byte[] convertedBytes = CharArrays.toUtf8Bytes(valueChars);
        assertArrayEquals(expectedBytes, convertedBytes);
    }

    public void testBytesToUtf8Chars() {
        final String originalValue = randomUnicodeOfCodepointLengthBetween(0, 32);
        final byte[] bytes = originalValue.getBytes(StandardCharsets.UTF_8);
        final char[] expectedChars = originalValue.toCharArray();

        final char[] convertedChars = CharArrays.utf8BytesToChars(bytes);
        assertArrayEquals(expectedChars, convertedChars);
    }

    public void testCharsBeginsWith() {
        assertFalse(CharArrays.charsBeginsWith(randomAlphaOfLength(4), null));
        assertFalse(CharArrays.charsBeginsWith(null, null));
        assertFalse(CharArrays.charsBeginsWith(null, randomAlphaOfLength(4).toCharArray()));
        final String undesiredPrefix = randomAlphaOfLength(2);
        assertFalse(CharArrays.charsBeginsWith(undesiredPrefix, randomAlphaOfLengthNotBeginningWith(undesiredPrefix, 3, 8)));

        final String prefix = randomAlphaOfLengthBetween(2, 4);
        assertTrue(CharArrays.charsBeginsWith(prefix, prefix.toCharArray()));
        final char[] prefixedValue = prefix.concat(randomAlphaOfLengthBetween(1, 12)).toCharArray();
        assertTrue(CharArrays.charsBeginsWith(prefix, prefixedValue));

        final String modifiedPrefix = randomBoolean() ? prefix.substring(1) : prefix.substring(0, prefix.length() - 1);
        char[] nonMatchingValue;
        do {
            nonMatchingValue = modifiedPrefix.concat(randomAlphaOfLengthBetween(0, 12)).toCharArray();
        } while (new String(nonMatchingValue).startsWith(prefix));
        assertFalse(CharArrays.charsBeginsWith(prefix, nonMatchingValue));
        assertTrue(CharArrays.charsBeginsWith(modifiedPrefix, nonMatchingValue));
    }

    public void testConstantTimeEquals() {
        final String value = randomAlphaOfLengthBetween(0, 32);
        assertTrue(CharArrays.constantTimeEquals(value, value));
        assertTrue(CharArrays.constantTimeEquals(value.toCharArray(), value.toCharArray()));

        // we want a different string, so ensure the first character is different, but the same overall length
        final int length = value.length();
        final String other = length > 0 ? new String(randomAlphaOfLengthNotBeginningWith(value.substring(0, 1), length, length)) : "";
        final boolean expectedEquals = length == 0;

        assertThat("value: " + value + ", other: " + other, CharArrays.constantTimeEquals(value, other), is(expectedEquals));
        assertThat(CharArrays.constantTimeEquals(value.toCharArray(), other.toCharArray()), is(expectedEquals));
    }

    private char[] randomAlphaOfLengthNotBeginningWith(String undesiredPrefix, int min, int max) {
        char[] nonMatchingValue;
        do {
            nonMatchingValue = randomAlphaOfLengthBetween(min, max).toCharArray();
        } while (new String(nonMatchingValue).startsWith(undesiredPrefix));
        return nonMatchingValue;
    }
}
