/*
 * Copyright (C) 2013-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <wtf/text/WYHash.h>

#include <cstdio>

namespace TestWebKitAPI {

static const unsigned expected[256] = {
    12444761, 2520368, 11291191, 12313020, 4102770, 9220562, 11704657, 13797327, 8866282, 12773540, 6851378, 14462743, 6052993, 1650779,
    9354444, 12227900, 10238407, 14378667, 10867698, 5845601, 3708420, 10223351, 5991036, 4422666, 3864312, 7185620, 5649988, 5969606, 8028492,
    4306324, 5301074, 13253498, 8365090, 3154802, 14971702, 16041699, 730894, 10353939, 6278444, 3451215, 14553569, 10197447, 11756713, 7518762,
    465926, 4708726, 9921370, 14776684, 11495881, 14407064, 3452648, 13626038, 4281815, 1430771, 8849920, 11802132, 12306378, 6861310, 12744155,
    5527768, 644470, 16474463, 6240947, 6203427, 2255717, 8336075, 15850000, 2117619, 5262326, 3630944, 11138751, 5329615, 6215623, 13648280,
    11961183, 6990582, 15454963, 5527963, 6248453, 15920109, 1613050, 4831307, 381277, 15140263, 6317256, 6586871, 4376705, 16128166, 3418365,
    14577744, 13351190, 405250, 14344827, 2757042, 11610433, 3737159, 13404806, 7354374, 14639148, 12116958, 463000, 1077214, 12110471, 9740240,
    14420969, 10231153, 13195885, 3310730, 9906900, 6073044, 4913512, 6779191, 13345794, 9126997, 8291645, 5140699, 12041398, 7378858, 5883770,
    4512235, 2211704, 10519479, 4300361, 1433878, 9924924, 14849441, 12036961, 11151930, 10224746, 5255374, 15027576, 197135, 7445860, 1426754,
    7057777, 2915889, 15546943, 10289791, 368934, 11844307, 8422846, 13998238, 16181361, 6233492, 4639948, 3363414, 7309317, 4796452, 8037020,
    12543547, 16151630, 8721536, 3371534, 8392904, 4177418, 11034761, 15995940, 7360699, 5460232, 16692788, 8837552, 7567532, 13175547, 4526239,
    1736985, 16122382, 10885225, 4043284, 7688459, 736745, 10061062, 11289357, 3727923, 207511, 6146952, 13689524, 15864704, 8497930, 5657460,
    12407276, 3382138, 14202418, 9392972, 8871135, 3516413, 1588952, 5569049, 11560049, 16456977, 16271567, 6814510, 10627621, 16037143,
    2505611, 16289108, 11007849, 114492, 348309, 891080, 13743067, 9242458, 7805067, 14857911, 2810812, 13337148, 12893457, 7344408, 13098107,
    1759206, 6192739, 11619610, 6196103, 9465816, 5676118, 5182247, 1944465, 14639529, 14294642, 72415, 16086717, 15645374, 14970357, 3064412,
    15520543, 14953876, 3235463, 6728650, 11433489, 5587784, 13141380, 631078, 12951785, 6309493, 11439258, 4910663, 2077306, 6259187, 10299855,
    14911779, 15661416, 4632138, 7628262, 11969946, 1779844, 3781248, 2278734, 13025137, 7458664, 14844377, 11652316, 16059741, 12584584,
    11419202, 8783024, 14105942, 4562623
};

TEST(WTF, WYHasher)
{
    auto generateLatin1Array = [&](size_t size) {
        auto array = std::unique_ptr<Latin1Character[]>(new Latin1Character[size]);
        for (size_t i = 0; i < size; i++)
            array[i] = i;
        return array;
    };

    auto generateUTF16Array = [&](size_t size) {
        auto array = std::unique_ptr<char16_t[]>(new char16_t[size]);
        for (size_t i = 0; i < size; i++)
            array[i] = i;
        return array;
    };

    unsigned max8Bit = std::numeric_limits<uint8_t>::max();
    for (size_t size = 0; size <= max8Bit; size++) {
        std::unique_ptr<const Latin1Character[]> arr1 = generateLatin1Array(size);
        std::unique_ptr<const char16_t[]> arr2 = generateUTF16Array(size);
        unsigned left = WYHash::computeHashAndMaskTop8Bits(std::span { arr1.get(), size });
        unsigned right = WYHash::computeHashAndMaskTop8Bits(std::span { arr2.get(), size });
        ASSERT_EQ(left, right);
        ASSERT_EQ(left, expected[size]);
    }
}

} // namespace TestWebKitAPI
