{%MainUnit castlefonts.pas}
{
  Copyright 2001-2022 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{$ifdef read_interface}

type
  { Font loaded from a font file, like ttf or otf.
    This class is typically used for outline (scalable, vector) fonts in ttf or otf formats.
    But it can really deal with any font supported by
    the @url(https://www.freetype.org/ FreeType library), even bitmap fonts,
    see @url(https://www.freetype.org/freetype2/docs/ft2faq.html#general-what the summary of font formats supported by FreeType).

    This can load a font file, or it can use ready data in TTextureFontData.
    The latter allows to use this for fonts embedded in a Pascal source code,
    since our texture-font-to-pascal can convert a font file to a unit that defines
    ready TTextureFontData instance. }
  TCastleFont = class(TCastleAbstractFont)
  strict private
    FFont: TTextureFontData; //< nil when no font loaded successfully
    FOwnsFont: boolean;
    DrawableImage: TDrawableImage;
    GlyphsScreenRects, GlyphsImageRects: TFloatRectangleList;
    FURL: String;
    FOptimalSize: Cardinal;
    FAntiAliased: Boolean;
    FLoadBasicCharacters: Boolean;
    FLoadCharacters: String;
    function GetSmoothScaling: boolean;
    function GetScale: Single;
    procedure SetScale(const AValue: Single);
    procedure SetURL(const Value: String);
    procedure SetOptimalSize(const Value: Cardinal);
    procedure SetAntiAliased(const Value: Boolean);
    procedure SetLoadBasicCharacters(const Value: Boolean);
    procedure SetLoadCharacters(const Value: String);
    { Free stuff loaded by @link(Load). }
    procedure Unload;
    procedure Reload;
  strict protected
    procedure SetSize(const Value: Single); override;
    procedure GLContextClose; override;
  public
    const
      DefaultOptimalSize = 20;

    constructor Create(AOwner: TComponent); overload; override;
    destructor Destroy; override;
    function PropertySections(const PropertyName: String): TPropertySections; override;
    function FontLoaded: Boolean; overload; override;

    constructor Create(const URL: String;
      const ASize: Cardinal; const AnAntiAliased: Boolean;
      const ACharacters: TUnicodeCharList = nil); reintroduce; overload; deprecated 'use Create(Owner: TComponent), then assign properties to load font: OptimalSize, LoadCharacters, AntiAliased, URL';
    constructor Create(const URL: String;
      const ASize: Cardinal; const AnAntiAliased: Boolean;
      const ACharacters: TSetOfChars); reintroduce; overload; deprecated 'use Create(Owner: TComponent), then assign properties to load font: OptimalSize, LoadCharacters, AntiAliased, URL';

    { Load by reading a FreeType font file, like ttf.

      Providing charaters list as @nil means that we only create glyphs
      for SimpleAsciiCharacters, which includes only the basic ASCII characters.
      The ACharacters instance @italic(does not) become owned by this object,
      so remember to free it after calling this constructor.

      Loading a font data also changes @link(Size) to the underlying
      (optimal to render) font data size. }
    procedure Load(const URL: String;
      const ASize: Cardinal; const AnAntiAliased: Boolean;
      const ACharacters: TUnicodeCharList = nil;
      const AdjustProperties: Boolean = true); overload; deprecated 'assign properties to load font: OptimalSize, LoadCharacters, AntiAliased, URL';

    { Load from a ready TTextureFontData instance.
      @param(Data TTextureFontData instance containing loaded image
        and glyphs parameters.)
      @param(OwnsData If @true, the Data instance becomes owned
        by this class (will be freed in our constructor).
        Usually you @italic(do not) want this, since usually you pass Data
        from a unit generated by texture-font-to-pascal. In this case,
        the finalization of CastleTextureFont_Xxx unit will already free
        the TTextureFontData instance.)
      @param(AdjustProperties Adjust URL, OptimalSize, Size, AntiAliased
        to reflect loaded Data. Or reset them to reflect empty URL
        (and the rest at defaults) when Data = nil.) }
    procedure Load(const Data: TTextureFontData;
      const OwnsData: Boolean = false;
      const AdjustProperties: Boolean = true); overload;

    procedure PrepareResources; override;
    procedure Print(const X, Y: Single; const Color: TCastleColor;
      const S: string); override;
    function TextWidth(const S: string): Single; override;
    function TextHeight(const S: string): Single; override;
    function TextHeightBase(const S: string): Single; override;
    function TextMove(const S: string): TVector2; override;

    { Underlying font data. }
    property FontData: TTextureFontData read FFont; {$ifdef FPC}deprecated 'you should not need to use this directly';{$endif} // can't be removed from delphi because x3dnodes_standard_text.inc uses that

    { Scale applied to the rendered font, compared to @link(FontData).Size.
      Changing this is equivalent to changing the Size property. }
    property Scale: Single read GetScale write SetScale;
  published
    { Loaded font file.
      Typically this is used for outline (scalable, vector) fonts in ttf or otf formats.
      But it can really deal with any font format supported by
      the @url(https://www.freetype.org/ FreeType library), even bitmap fonts,
      see @url(https://www.freetype.org/freetype2/docs/ft2faq.html#general-what the summary of font formats supported by FreeType). }
    property URL: String read FURL write SetURL;

    { Optimal font size (in real device pixels),
      the font will be scaled when other size is actually needed.
      This also sets default @link(Size) used for rendering this font. }
    property OptimalSize: Cardinal read FOptimalSize write SetOptimalSize default DefaultOptimalSize;

    { Anti-aliased font has smooth edges and is rendered using blending.
      Usually this is much better for quality.
      Non-anti-aliased means that font uses simple yes/no transparency
      and is rendered using alpha testing. }
    property AntiAliased: Boolean read FAntiAliased write SetAntiAliased default true;

    { Load from font all "basic" characters, which include digits,
      English letters and standard ASCII symbols.
      See @link(SimpleAsciiCharacters) for exact definition.
      These are loaded in addition to characters listed on LoadCharacters. }
    property LoadBasicCharacters: Boolean read FLoadBasicCharacters write SetLoadBasicCharacters default true;

    { Load from font all characters listed here. As everywhere in CGE
      (same as in Lazarus LCL), this is an string composed from UTF-8 characters. }
    property LoadCharacters: String read FLoadCharacters write SetLoadCharacters;
  end;

  TGLBitmapFont = TCastleFont deprecated 'use TCastleFont';
  TTextureFont = TCastleFont deprecated 'use TCastleFont';

{$endif read_interface}

{$ifdef read_implementation}

{ TCastleFont --------------------------------------------------------------- }

constructor TCastleFont.Create(AOwner: TComponent);
begin
  inherited;
  GlyphsScreenRects := TFloatRectangleList.Create;
  GlyphsImageRects  := TFloatRectangleList.Create;
  FLoadBasicCharacters := true;
  FOptimalSize := DefaultOptimalSize;
  FAntiAliased := true;
end;

destructor TCastleFont.Destroy;
begin
  Unload;
  FreeAndNil(GlyphsScreenRects);
  FreeAndNil(GlyphsImageRects);
  inherited;
end;

procedure TCastleFont.Unload;
begin
  // free previous FFont data
  Load(nil, { OwnsData doesn't matter } false, { AdjustParameters } false);
end;

constructor TCastleFont.Create(const URL: String;
  const ASize: Cardinal; const AnAntiAliased: Boolean;
  const ACharacters: TUnicodeCharList);
begin
  Create(nil);
  {$warnings off} // calling deprecated from deprecated
  Load(URL, ASize, AnAntiAliased, ACharacters, true);
  {$warnings on}
end;

procedure TCastleFont.Load(const URL: String;
  const ASize: Cardinal; const AnAntiAliased: Boolean;
  const ACharacters: TUnicodeCharList = nil;
  const AdjustProperties: Boolean = true);
var
  NewFontData: TTextureFontData;
begin
  { Although FontSizesChanged is also called by Load,
    make sure it gets called even when TTextureFontData.Create fails
    (e.g. loading file fails). }
  FontSizesChanged;
  if URL = '' then
    Unload
  else
  begin
    NewFontData := TTextureFontData.Create(URL, ASize, AnAntiAliased, ACharacters);
    Load(NewFontData, true, AdjustProperties);
  end;
end;

constructor TCastleFont.Create(const URL: String;
  const ASize: Cardinal; const AnAntiAliased: Boolean;
  const ACharacters: TSetOfChars);
var
  Chars: TUnicodeCharList;
  C: char;
begin
  Chars := TUnicodeCharList.Create;
  try
    for C in ACharacters do
      Chars.Add(Ord(C));
    {$warnings off} // calling deprecated from deprecated
    Create(URL, ASize, AnAntiAliased, Chars);
    {$warnings on}
  finally FreeAndNil(Chars) end;
end;

procedure TCastleFont.Load(const Data: TTextureFontData; const OwnsData: Boolean = false;
  const AdjustProperties: Boolean = true);
begin
  GLContextClose;

  if FOwnsFont then
    FreeAndNil(FFont)
  else
    FFont := nil;

  FOwnsFont := OwnsData;
  FFont := Data;

  if AdjustProperties then
  begin
    if FFont <> nil then
    begin
      FURL := FFont.URL;
      FOptimalSize := FFont.Size;
      FAntiAliased := FFont.AntiAliased;
    end else
    begin
      FURL := '';
      { for consistency, adjust all properties always, even when FFont = nil }
      FOptimalSize := DefaultOptimalSize;
      FAntiAliased := true;
    end;
    // synchronize Size with OptimalSize
    Size := OptimalSize;
  end;

  // call this at the end, when our state is consistent
  FontSizesChanged;
end;

function TCastleFont.PropertySections(const PropertyName: String): TPropertySections;
begin
  if (PropertyName = 'URL') or
     (PropertyName = 'OptimalSize') or
     (PropertyName = 'AntiAliased') or
     (PropertyName = 'LoadCharacters') or
     (PropertyName = 'LoadBasicCharacters') then
    Result := [psBasic]
  else
    Result := inherited PropertySections(PropertyName);
end;

function TCastleFont.GetScale: Single;
begin
  if FFont <> nil then
    Result := Size / FFont.Size
  else
    Result := Size;
end;

procedure TCastleFont.SetScale(const AValue: Single);
begin
  if FFont <> nil then
    Size := FFont.Size * AValue
  else
    Size := AValue;
end;

procedure TCastleFont.SetSize(const Value: Single);
begin
  inherited SetSize(Value);

  Assert((FFont = nil) or (FFont.Size <> 0));
  Assert(not IsInfinite(Value));

  if DrawableImage <> nil then
    DrawableImage.SmoothScaling := GetSmoothScaling;
end;

function TCastleFont.GetSmoothScaling: boolean;
begin
  Assert(FFont <> nil); // should never be called otherwise
  Result := Size <> FFont.Size;
end;

procedure TCastleFont.PrepareResources;
begin
  inherited;
  if (FFont <> nil) and (DrawableImage = nil) then
    DrawableImage := TDrawableImage.Create(FFont.Image, GetSmoothScaling, false);
end;

procedure TCastleFont.GLContextClose;
begin
  FreeAndNil(DrawableImage);
  inherited;
end;

procedure TCastleFont.Print(const X, Y: Single; const Color: TCastleColor;
  const S: string);
var
  ScreenX, ScreenY: Single;
  G: TTextureFontData.TGlyph;
  GlyphsToRender: Integer;

  procedure GlyphDraw(const OutlineMoveX, OutlineMoveY: Integer);
  var
    ScreenRect, ImageRect: PFloatRectangle;
  begin
    if TargetImage <> nil then
    begin
      TargetImage.DrawFrom(FFont.Image,
        Round(ScreenX - G.X * Scale + OutlineMoveX * Outline),
        Round(ScreenY - G.Y * Scale + OutlineMoveY * Outline),
        G.ImageX,
        G.ImageY,
        G.Width,
        G.Height);
    end else
    begin
      Assert(GlyphsToRender < GlyphsScreenRects.Count);

      { Use a small margin around every glyph to allow bilinear
        filtering to smoothly go from opaque to fully transparent
        at glyph border. This prevents glyph border from ending suddenly,
        it looks much better in the case of blending.

        This cooperates with TTextureFontData.Create (used by
        texture-font-to-pascal) that makes sure that each letter is
        surrounded with a padding that allows such border,
        see GlyphPadding in castletexturefontdata.pas. }
      { TODO; Not used now. While there's a visible improvement in some cases
        (when optimal font size mismatches final font size),
        there's also regression in view3dscene
        (when optimal font size matches final font size closely):
        the "A" in "Animations" button in view3dscene gets weird artifact. }
      {.$define EXTRA_GLYPH_SPACE}

      ScreenRect := PFloatRectangle(GlyphsScreenRects.Ptr(GlyphsToRender));
      ScreenRect^.Left   := ScreenX - (G.X {$ifdef EXTRA_GLYPH_SPACE} + 0.5 {$endif}) * Scale + OutlineMoveX * Outline;
      ScreenRect^.Bottom := ScreenY - (G.Y {$ifdef EXTRA_GLYPH_SPACE} + 0.5 {$endif}) * Scale + OutlineMoveY * Outline;
      ScreenRect^.Width  := (G.Width  {$ifdef EXTRA_GLYPH_SPACE} + 1 {$endif}) * Scale;
      ScreenRect^.Height := (G.Height {$ifdef EXTRA_GLYPH_SPACE} + 1 {$endif}) * Scale;

      ImageRect := PFloatRectangle(GlyphsImageRects.Ptr(GlyphsToRender));
      ImageRect^.Left   := G.ImageX {$ifdef EXTRA_GLYPH_SPACE} - 0.5 {$endif};
      ImageRect^.Bottom := G.ImageY {$ifdef EXTRA_GLYPH_SPACE} - 0.5 {$endif};
      ImageRect^.Width  := G.Width  {$ifdef EXTRA_GLYPH_SPACE} + 1 {$endif};
      ImageRect^.Height := G.Height {$ifdef EXTRA_GLYPH_SPACE} + 1 {$endif};

      Inc(GlyphsToRender);
    end;
  end;

var
  C: TUnicodeChar;
  {$ifdef FPC}
  TextPtr: PChar;
  CharLen: Integer;
  {$else}
  TextIndex: Integer;
  NextTextIndex: Integer;
  TextLength: Integer;
  {$endif}
  GlyphsPerChar: Integer;
begin
  if FFont = nil then
    Exit; // font not loaded

  if TargetImage = nil then
  begin
    PrepareResources;

    { allocate the necessary glyphs at start.
      This allows to quickly fill them later.
      Note that we possibly allocate too much, because Length(S) may be > UTF8Length(S)
      (because of multi-byte characters), and also because some characters do not have glyphs.
      That's OK, we'll calculate real GlyphsToRender when iterating. }
    if Outline = 0 then
      GlyphsPerChar := 1 else
    if OutlineHighQuality then
      GlyphsPerChar := 8 else
      GlyphsPerChar := 4;
    GlyphsScreenRects.Count := Max(MinimumGlyphsAllocated, GlyphsPerChar * Length(S));
    GlyphsImageRects .Count := Max(MinimumGlyphsAllocated, GlyphsPerChar * Length(S));
  end;

  { first pass, to render Outline.

    This could be done better by being done together with non-outline pass,
    by filling the alternative place in Glyph arrays, such that outline and non-outline data
    don't collide.
    It would be 1. faster (don't iterate over S two times), 2. less code duplication. }
  if Outline <> 0 then
  begin
    GlyphsToRender := 0;
    { While Round() below is not needed, it improves the quality of rendered
      text. Compare e.g. view3dscene button captions. }
    ScreenX := Round(X);
    ScreenY := Round(Y);
    if TargetImage <> nil then
      FFont.Image.ColorWhenTreatedAsAlpha := Vector3Byte(OutlineColor.XYZ); // ignore OutlineColor[3] for now

    {$ifdef FPC}
    TextPtr := PChar(S);
    C := UTF8CharacterToUnicode(TextPtr, CharLen);
    while (C > 0) and (CharLen > 0) do
    {$else}
    TextIndex := 1;
    TextLength := Length(S);
    while (TextIndex <= TextLength) do
    {$endif}
    begin
      {$ifdef FPC}
      Inc(TextPtr, CharLen);
      {$else}
      C := GetUTF32Char(S, TextIndex, NextTextIndex);
      TextIndex := NextTextIndex;
      {$endif}

      G := FFont.Glyph(C);
      if G <> nil then
      begin
        if (G.Width <> 0) and (G.Height <> 0) then
        begin
          GlyphDraw(0, 0);
          GlyphDraw(0, 2);
          GlyphDraw(2, 2);
          GlyphDraw(2, 0);

          if OutlineHighQuality then
          begin
            GlyphDraw(1, 0);
            GlyphDraw(1, 2);
            GlyphDraw(0, 1);
            GlyphDraw(2, 1);
          end;
        end;
        ScreenX := ScreenX + G.AdvanceX * Scale + Outline * 2;
        ScreenY := ScreenY + G.AdvanceY * Scale;
      end;

      {$ifdef FPC}
      C := UTF8CharacterToUnicode(TextPtr, CharLen);
      {$endif}
    end;

    if TargetImage = nil then
    begin
      DrawableImage.Color := OutlineColor;
      DrawableImage.Draw(
        PFloatRectangleArray(GlyphsScreenRects.List),
        PFloatRectangleArray(GlyphsImageRects.List), GlyphsToRender);
    end;
  end;

  GlyphsToRender := 0;
  { While Round() below is not needed, it improves the quality of rendered
    text. }
  ScreenX := Round(X);
  ScreenY := Round(Y);
  if TargetImage <> nil then
    FFont.Image.ColorWhenTreatedAsAlpha := Vector3Byte(Color.XYZ); // ignore Color[3] for now

  {$ifdef FPC}
  TextPtr := PChar(S);
  C := UTF8CharacterToUnicode(TextPtr, CharLen);
  while (C > 0) and (CharLen > 0) do
  {$else}
  TextIndex := 1;
  TextLength := Length(S);
  while (TextIndex <= TextLength) do
  {$endif}
  begin
    {$ifdef FPC}
    Inc(TextPtr, CharLen);
    {$else}
    C := GetUTF32Char(S, TextIndex, NextTextIndex);
    TextIndex := NextTextIndex;
    {$endif}

    G := FFont.Glyph(C);
    if G <> nil then
    begin
      if (G.Width <> 0) and (G.Height <> 0) then
        if Outline <> 0 then
          GlyphDraw(1, 1)
        else
          GlyphDraw(0, 0);
      ScreenX := ScreenX + G.AdvanceX * Scale + Outline * 2;
      ScreenY := ScreenY + G.AdvanceY * Scale;
    end;

    {$ifdef FPC}
    C := UTF8CharacterToUnicode(TextPtr, CharLen);
    {$endif}
  end;

  if TargetImage = nil then
  begin
    DrawableImage.Color := Color;
    DrawableImage.Draw(
      PFloatRectangleArray(GlyphsScreenRects.List),
      PFloatRectangleArray(GlyphsImageRects.List), GlyphsToRender);
  end;
end;

function TCastleFont.TextWidth(const S: string): Single;
begin
  if FFont = nil then
    Exit(0); // font not loaded
  Result := FFont.TextWidth(S) * Scale;
  if Outline <> 0 then
    Result := Result + Outline * 2 * {$ifdef FPC}UTF8Length{$else}GetUTF32Length{$endif}(S);
end;

function TCastleFont.TextHeight(const S: string): Single;
begin
  if FFont = nil then
    Exit(0); // font not loaded
  Result := FFont.TextHeight(S) * Scale + Outline * 2;
end;

function TCastleFont.TextHeightBase(const S: string): Single;
begin
  if FFont = nil then
    Exit(0); // font not loaded
  Result := FFont.TextHeightBase(S) * Scale + Outline * 2;
end;

function TCastleFont.TextMove(const S: string): TVector2;
var
  M: TVector2Integer;
begin
  if FFont = nil then
    Exit(TVector2.Zero); // font not loaded
  M := FFont.TextMove(S);
  Result := Vector2(M.X, M.Y);
  Result.X := Result.X * Scale;
  if Outline <> 0 then
    Result.X := Result.X + Outline * 2 * {$ifdef FPC}UTF8Length{$else}GetUTF32Length{$endif}(S);
  Result.Y := Result.Y * Scale;
end;

procedure TCastleFont.SetURL(const Value: String);
begin
  if FURL <> Value then
  begin
    { We deliberately change URL even when loading of font file could fail.
      This way URL property is always correctly deserialized in CGE editor,
      even when loading the image failed. }
    FURL := Value;
    Reload;
  end;
end;

procedure TCastleFont.SetOptimalSize(const Value: Cardinal);
begin
  if FOptimalSize <> Value then
  begin
    { As with SetURL: we deliberately change current property, even when Reload fails. }
    FOptimalSize := Value;
    Reload;
  end;
end;

procedure TCastleFont.SetAntiAliased(const Value: Boolean);
begin
  if FAntiAliased <> Value then
  begin
    { As with SetURL: we deliberately change current property, even when Reload fails. }
    FAntiAliased := Value;
    Reload;
  end;
end;

procedure TCastleFont.SetLoadBasicCharacters(const Value: Boolean);
begin
  if FLoadBasicCharacters <> Value then
  begin
    { As with SetURL: we deliberately change current property, even when Reload fails. }
    FLoadBasicCharacters := Value;
    Reload;
  end;
end;

procedure TCastleFont.SetLoadCharacters(const Value: String);
begin
  if FLoadCharacters <> Value then
  begin
    { As with SetURL: we deliberately change current property, even when Reload fails. }
    FLoadCharacters := Value;
    Reload;
  end;
end;

procedure TCastleFont.Reload;

  function GetCharactersList: TUnicodeCharList;
  begin
    if (LoadCharacters = '') and LoadBasicCharacters then
    begin
      { Default, and fast case. TTextureFontData.Create interprets characters list = nil
        just like that: load only SimpleAsciiCharacters.
        No need to create TUnicodeCharList instance to express it explicitly. }
      Result := nil;
    end else
    begin
      Result := TUnicodeCharList.Create;
      if LoadBasicCharacters then
        Result.Add(SimpleAsciiCharacters);
      Result.Add(LoadCharacters);
    end;
  end;

  // Load, without catching any exceptions
  procedure LoadCore;
  var
    CharactersList: TUnicodeCharList;
  begin
    CharactersList := GetCharactersList;
    try
      {$warnings off} // this Load is deprecated; actually it should be internal, nested here
      Load(URL, OptimalSize, AntiAliased, CharactersList);
      {$warnings on}
    finally FreeAndNil(CharactersList) end;
  end;

begin
  if CastleDesignMode then
  begin
    { If loading file failed, and we're inside CGE editor,
      merely report a warning. This allows deserializing in CGE editor
      designs with broken URLs. }
    try
      LoadCore;
    except
      on E: Exception do
      begin
        { We unload, otherwise we would keep the font loaded, not correctly reflecting
          some new properties.
          Testcase: in editor load TTF to some TCastleFont,
          use it with some TCastleLabel.CustomFont,
          then set TCastlFont.LoadBasicCharacters to false -> font should revert
          to default UIFont (with warning "cannot load font with no glyphs"). }
        Unload;
        WritelnWarning('TCastleFont', 'Failed to load font "%s": %s',
          [URIDisplay(URL), ExceptMessage(E)]);
      end;
    end;
  end else
  begin
    { When not in editor, most exceptions are just passed upward.
      But we handle EFreeTypeLibraryNotFound. }
    try
      LoadCore;
    except
      on E: EFreeTypeLibraryNotFound do
      begin
        Unload;
        WritelnWarning('TCastleFont', 'Failed to load font "%s": %s',
          [URIDisplay(URL), E.Message]);
      end;
    end;
  end;
end;

function TCastleFont.FontLoaded: Boolean;
begin
  Result := FFont <> nil;
end;

{$endif read_implementation}
