# This file is part of Tryton.  The COPYRIGHT file at the top level of
# this repository contains the full copyright notices and license terms.
from itertools import chain
from functools import wraps

from trytond.model import ModelView, Workflow, fields
from trytond.pool import Pool, PoolMeta
from trytond.transaction import Transaction

__all__ = ['Invoice', 'InvoiceLine']


def process_purchase(func):
    @wraps(func)
    def wrapper(cls, invoices):
        pool = Pool()
        Purchase = pool.get('purchase.purchase')
        with Transaction().set_context(_check_access=False):
            purchases = set(
                p for i in cls.browse(invoices) for p in i.purchases)
        func(cls, invoices)
        if purchases:
            Purchase.__queue__.process(purchases)
    return wrapper


class Invoice(metaclass=PoolMeta):
    __name__ = 'account.invoice'
    purchase_exception_state = fields.Function(fields.Selection([
        ('', ''),
        ('ignored', 'Ignored'),
        ('recreated', 'Recreated'),
        ], 'Exception State'), 'get_purchase_exception_state')
    purchases = fields.Function(fields.One2Many('purchase.purchase', None,
            'Purchases'), 'get_purchases', searcher='search_purchases')

    @classmethod
    def __setup__(cls):
        super(Invoice, cls).__setup__()
        cls._error_messages.update({
                'reset_invoice_purchase': ('You cannot reset to draft '
                    'an invoice generated by a purchase.'),
                })

    def get_purchase_exception_state(self, name):
        purchases = self.purchases

        recreated = tuple(i for p in purchases for i in p.invoices_recreated)
        ignored = tuple(i for p in purchases for i in p.invoices_ignored)

        if self in recreated:
            return 'recreated'
        elif self in ignored:
            return 'ignored'
        return ''

    def get_purchases(self, name):
        pool = Pool()
        PurchaseLine = pool.get('purchase.line')
        purchases = set()
        for line in self.lines:
            if isinstance(line.origin, PurchaseLine):
                purchases.add(line.origin.purchase.id)
        return list(purchases)

    @classmethod
    def search_purchases(cls, name, clause):
        return [('lines.origin.purchase' + clause[0].lstrip(name),)
            + tuple(clause[1:3]) + ('purchase.line',) + tuple(clause[3:])]

    @classmethod
    @process_purchase
    def delete(cls, invoices):
        super(Invoice, cls).delete(invoices)

    @classmethod
    def copy(cls, invoices, default=None):
        if default is None:
            default = {}
        else:
            default = default.copy()
        default.setdefault('purchases', None)
        return super(Invoice, cls).copy(invoices, default=default)

    @classmethod
    @ModelView.button
    @Workflow.transition('draft')
    def draft(cls, invoices):
        purchases = list(chain(*(i.purchases for i in invoices)))
        if purchases and any(i.state == 'cancel' for i in invoices):
            cls.raise_user_error('reset_invoice_purchase')

        return super(Invoice, cls).draft(invoices)

    @classmethod
    @process_purchase
    def post(cls, invoices):
        super(Invoice, cls).post(invoices)

    @classmethod
    @process_purchase
    def paid(cls, invoices):
        super(Invoice, cls).paid(invoices)

    @classmethod
    @process_purchase
    def cancel(cls, invoices):
        super(Invoice, cls).cancel(invoices)


class InvoiceLine(metaclass=PoolMeta):
    __name__ = 'account.invoice.line'

    @property
    def origin_name(self):
        pool = Pool()
        PurchaseLine = pool.get('purchase.line')
        name = super(InvoiceLine, self).origin_name
        if isinstance(self.origin, PurchaseLine):
            name = (self.origin.purchase.reference
                or self.origin.purchase.rec_name)
        return name

    @classmethod
    def _get_origin(cls):
        models = super(InvoiceLine, cls)._get_origin()
        models.append('purchase.line')
        return models

    @classmethod
    def delete(cls, lines):
        pool = Pool()
        Purchase = pool.get('purchase.purchase')
        with Transaction().set_context(_check_access=False):
            invoices = (l.invoice for l in cls.browse(lines)
                if l.type == 'line' and l.invoice)
            purchases = set(p for i in invoices for p in i.purchases)
        super(InvoiceLine, cls).delete(lines)
        if purchases:
            Purchase.__queue__.process(purchases)
