/*
 * @(#)xmball.c
 *
 * Copyright 1994 - 2024  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Mball */

#ifndef WINVER
#include "version.h"
static const char aboutHelp[] = {
"Mball Version " VERSION "\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd AT verizon.net>\n"
"The latest version is at: "
"https://www.sillycycle.com/puzzles.html"
};

static const char synopsisHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-{border|bd} {color}]\n"
"[-wedge{0|1|2|3|4|5|6|7|8|9|10|11} {color}]\n"
"[-frame {color}] [-delay msecs] [-[no]sound]\n"
"[-moveSound {filename}] [-{font|fn} {fontname}]\n"
"[-view {int}] [-[no]perspective] [-wedges {int}]\n"
"[-bands {int}] [-[no]orient] [-[no]practice] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA) || defined(WINVER)
static const char descriptionHelp[] = {
"The original puzzle has 8 wedges on a sphere "
"(longitudinal cuts), with each wedge divided into 4 bands\n"
"(latitudinal cuts).  There are essentially 3 varieties: "
"Geomaster 8 colors (beachball, default colors in this\n"
"puzzle), Duomaster 2 colors (black and white beachball), "
"and a variety of picture Masterballs.  For some reason,\n"
"they say the Geomaster is easier than the Duomaster.  The "
"picture Masterballs are the hardest since all the pieces\n"
"have a set solved position.  On the Duo and Geo Masterballs "
"pieces could be swapped or in a different order and still\n"
"be in a solved position.  Zurick University's Mathematics "
"faculty has calculated that Masterball's 32 segments can\n"
"be arranged in 355,682,548,566,633,480,192,000,000 different "
"possible combinations.  Masterball was invented by Dr.\n"
"Geza Gyovai Hungarian Engineer, manufactured by Whole "
"Systems Design, Inc..\n"
"\n"
"Related puzzles are the Saturn Puzzle, an oriented Masterball "
"with one band, and the harder Hockey Puck Puzzle, like\n"
"the Saturn Puzzle but with 12 wedges instead of 8."
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece and the pieces will\n"
"then turn towards where the mouse button was released.  "
"Usually, a click and release on the same wedge is\n"
"ambiguous and the puzzle will not turn.  If there is "
"some ambiguity between a turn or a flip, a flip is\n"
"assumed, except if shift key is used.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to read (get) a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"Press \"S\" or \"s\" keys to start auto-solver.  Only works "
"for wedges <= 8 or 1 band.\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
"has to orient the wedges in orient mode, besides\n"
"getting all the wedges to be the same color.  To do this "
"one has to get the numbers to be on the same side\n"
"of the ball in clockwise order.\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of bands.\n"
"\n"
"Press \"D\" or \"d\" keys to decrease the number of bands.\n"
"\n"
"Press \"2\", \"4\", \"6\", \"8\", \"0\", or \"=\" keys "
"(not the keypad 2, 4, 6, 8, 0) to change to 2, 4, 6, 8, 10,\n"
"or 12 wedges, respectively.  Note: if there were odd "
"number of wedges, there would be no 180 degree turn\n"
"and therefore the puzzle would be inoperable.\n"
"\n"
"Press \"V\" or \"v\" keys to change the view of the puzzle.\n"
"\n"
"Press \"T\" or \"t\" keys to toggle the perspective look.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of pieces.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of pieces.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Master Ball as:\n"
"N / *  Upper Upper Left, Counterclockwise, Upper Upper Right\n"
"7 8 9  Up, Upper Right\n"
"  ^\n"
"4<5>6  Left, Clockwise, Right\n"
"  v\n"
"1 2 3  Lower Left, Down, Lower Right\n"
" 0  .  Lower Lower Left, Lower Lower Right\n"
"\n"
"Use the control key and the left mouse button, keypad, "
"or arrow keys to move the whole Masterball.  This\n"
"is not recorded as a turn."
};

static const char referencesHelp[] = {
"http://wsd.com/masterball"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for wedges <= 8 or 1 band."
};

#include "file.h"
#ifdef WINVER
#include "MballP.h"
#define TITLE "wmball"

static MballRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->mball.started = b
#else
#include "xwin.h"
#include "xgui.h"
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Mball.h"
#include "Mball2d.h"
#ifdef HAVE_OPENGL
#include "MballGL.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "icons/16x16/mball.xpm"
#include "icons/22x22/mball.xpm"
#include "icons/24x24/mball.xpm"
#include "icons/32x32/mball.xpm"
#include "icons/48x48/mball.xpm"
#include "icons/64x64/mball.xpm"
#else
#include "pixmaps/16x16/mball.xpm"
#include "pixmaps/22x22/mball.xpm"
#include "pixmaps/24x24/mball.xpm"
#include "pixmaps/32x32/mball.xpm"
#include "pixmaps/48x48/mball.xpm"
#include "pixmaps/64x64/mball.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?mball_16x16:mball_22x22):\
(((s)<=24)?mball_24x24:mball_32x32)):\
(((s)<=48)?mball_48x48:mball_64x64)))
#endif
#include "pixmaps/64x64/mball.xbm"
#define DEFINE_XBM (char *) mball_64x64_bits, mball_64x64_width, mball_64x64_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "mball.scores"
#endif

#define MAX_BANDS 6
#define NEVER (-1)
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[2][(MAX_WEDGES - MIN_WEDGES) / 2 + 1]
[MAX_BANDS - MIN_BANDS + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static Widget movesText, recordText, message;
static Widget orientToggle, practiceToggle;
static Widget soundMenuItem, perspectiveMenuItem;
static Widget practiceDialog, randomizeDialog, solveDialog;
static Widget descriptionDialog, featuresDialog;
static Widget synopsisDialog, referencesDialog, aboutDialog;
static char buff[MESSAGE_LENGTH];
static const char *wedgesLabel = "Wedges:", *bandsLabel = "Bands:";
static const char *speedLabel = "Animation Speed:";
#define PRINT_MESSAGE(b) printState(message, b)
#define MIN_SPEED 1
#define MAX_SPEED 50
static const char *wedgeStrings[] =
{
	"Two", "Four", "Six", "Eight", "Ten", "Twelve"
};
#elif !defined(WINVER)
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#ifdef HAVE_MOTIF
static Widget wedgeChoices[(MAX_WEDGES - MIN_WEDGES) / 2 + 1];
static Widget bandChanger, speedChanger;
#elif defined(HAVE_ATHENA)
static Widget wedgeLabel = None;
#ifndef USE_POPUP
static Widget wedgeChoices[(MAX_WEDGES - MIN_WEDGES) / 2 + 1];
#endif
static Widget bandSlider, speedSlider;
static Widget bandSliderLabel, speedSliderLabel;
static const char *fileTypes[] =
{
	"Get",
	"Write",
	"Exit"
};
#define numFileTypes (sizeof(fileTypes)/sizeof(fileTypes[0]))
static const char *playTypes[] =
{
	"Undo",
	"Redo",
	"Clear",
#ifdef EXTRA
	"Practice",
#endif
	"Randomize",
	"Auto-solve",
#ifdef EXTRA
	"Increment Bands",
	"Decrement Bands",
	"Oriented",
	"Speed >",
	"Slow <",
#endif
	"Sound @",
	"View"
	"Perspective"
};
#define numPlayTypes (sizeof(playTypes)/sizeof(playTypes[0]))
static const char *helpTypes[] =
{
	"Description...",
	"Features...",
	"Synopsis...",
	"References...",
	"About..."
};
#define numHelpTypes (sizeof(helpTypes)/sizeof(helpTypes[0]))
#else
#if !defined(WINVER) && defined(HAVE_OPENGL)
static Widget shell;
#endif
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle2d;
#ifdef HAVE_OPENGL
static Widget puzzle3d;
#endif
static char userNameDsp[USER_NAME_LENGTH] = "";
#define WINDOW_HEIGHT 300

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w,
		XmNlabelString, xmstr, NULL);
	XmStringFree(xmstr);
}
#elif defined(HAVE_ATHENA)
static void
printState(Widget w, char *msg)
{
	XtVaSetValues(w,
		XtNlabel, msg, NULL);
	if (w == movesText)
		XtVaSetValues(w,
			XtNwidth, 64, NULL);
}
#endif

static void
printRecords(void)
{
	int i, j, orient;

	(void) printf("           MBALL  HALL OF FAME\n\n");
	(void) printf("ORIENT WEDGE BAND USER            MOVES\n");
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < (MAX_WEDGES - MIN_WEDGES) / 2 + 1; i++)
			for (j = 0; j < MAX_BANDS - MIN_BANDS + 1; j++) {
				if (puzzleRecord[orient][i][j].score > 0)
					(void) printf("%6d%6d%5d %-12s%9d\n",
						orient, 2 * i + 1, j + 1,
						puzzleRecord[orient][i][j].name,
						puzzleRecord[orient][i][j].score);
			}
}
#endif

static void
initRecords(void)
{
	int i, j, orient;

	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < (MAX_WEDGES - MIN_WEDGES) / 2 + 1; i++)
			for (j = 0; j < MAX_BANDS - MIN_BANDS + 1; j++) {
				puzzleRecord[orient][i][j].score = NEVER;
				(void) strncpy(puzzleRecord[orient][i][j].name,
					NOACCESS, USER_NAME_LENGTH);
			}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, j, n, orient;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Cannot read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (orient = 0; orient < 2; orient++)
		for (i = 0; i < (MAX_WEDGES - MIN_WEDGES) / 2 + 1; i++)
			for (j = 0; j < MAX_BANDS - MIN_BANDS + 1; j++) {
				if (fscanf(fp, "%d %s\n", &n, userName) != 2) {
					(void) fprintf(stderr,
						"corrupt record, expecting integer and name for %d %d %d\n",
						orient, i, j);
					(void) fclose(fp);
					return;
				}
				if (n <= puzzleRecord[orient][i][j].score ||
						puzzleRecord[orient][i][j].score <= NEVER) {
					puzzleRecord[orient][i][j].score = n;
					(void) strncpy(puzzleRecord[orient][i][j].name,
						userName, USER_NAME_LENGTH);
				}
			}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, j, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Cannot write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		lockFile[FILE_NAME_LENGTH - 6] = '\0';
		(void) strncat(lockFile, ".lock", 6);
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < (MAX_WEDGES - MIN_WEDGES) / 2 + 1; i++) {
				for (j = 0; j < MAX_BANDS - MIN_BANDS + 1; j++)
					(void) fprintf(fp, "%d %s\n",
						puzzleRecord[orient][i][j].score,
						puzzleRecord[orient][i][j].name);
				if ((MAX_WEDGES - MIN_WEDGES) / 2 ||
						orient < 1)
					(void) fprintf(fp, "\n");
			}
			if (orient < 1)
				(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int wedges, int bands, Boolean orient, Boolean practice)
{
	int i = (orient) ? 1 : 0;
	int w = (wedges - MIN_WEDGES) / 2;
	int r = bands - MIN_BANDS;

	if (practice) {
		(void) strncpy(recordDsp, "practice", 9);
	} else if (bands > MAX_BANDS) {
		(void) strncpy(recordDsp, "NOT RECORDED", 13);
	} else if (puzzleRecord[i][w][r].score <= NEVER) {
		(void) strncpy(recordDsp, "NEVER ", 7);
		(void) strncat(recordDsp, NOACCESS, strlen(NOACCESS) + 1);
	} else {
#ifdef HAVE_SNPRINTF
		(void) snprintf(recordDsp, MESSAGE_LENGTH, "%d %s",
			puzzleRecord[i][w][r].score,
			puzzleRecord[i][w][r].name);
#else
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[i][w][r].score,
			puzzleRecord[i][w][r].name);
#endif
	}
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(recordText, recordDsp);
#endif
}

#ifndef WINVER
/* There is probably a better way to assure that they are the same
 * but I do not know it off hand. */
static void
makeEquivalent(String *userName, String *scoreFile, int *wedges, int *bands,
		Boolean *orient, Boolean *practice, int *delay)
{
	Boolean mono, reverse;
	Boolean scoreOnly, versionOnly;
	Pixel foreground, background, pieceBorder;
	String wedgeColor[MAX_WEDGES];

	XtVaGetValues(puzzle2d,
		XtNuserName, userName,
		XtNscoreFile, scoreFile,
		XtNwedges, wedges,
		XtNbands, bands,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverseVideo, &reverse,
		XtNdelay, delay,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNwedgeColor0, &(wedgeColor[0]),
		XtNwedgeColor1, &(wedgeColor[1]),
		XtNwedgeColor2, &(wedgeColor[2]),
		XtNwedgeColor3, &(wedgeColor[3]),
		XtNwedgeColor4, &(wedgeColor[4]),
		XtNwedgeColor5, &(wedgeColor[5]),
		XtNwedgeColor6, &(wedgeColor[6]),
		XtNwedgeColor7, &(wedgeColor[7]),
		XtNwedgeColor8, &(wedgeColor[8]),
		XtNwedgeColor9, &(wedgeColor[9]),
		XtNwedgeColor10, &(wedgeColor[10]),
		XtNwedgeColor11, &(wedgeColor[11]),
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(*scoreFile, "")) {
		(void) strncpy(scoreFileName, *scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
	XtVaSetValues(puzzle2d,
		XtNdirection, IGNORE_DIR,
		XtNstart, False, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNuserName, *userName,
		XtNwedges, *wedges,
		XtNbands, *bands,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverseVideo, reverse,
		XtNdelay, *delay,
		XtNdirection, IGNORE_DIR,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNwedgeColor0, wedgeColor[0],
		XtNwedgeColor1, wedgeColor[1],
		XtNwedgeColor2, wedgeColor[2],
		XtNwedgeColor3, wedgeColor[3],
		XtNwedgeColor4, wedgeColor[4],
		XtNwedgeColor5, wedgeColor[5],
		XtNwedgeColor6, wedgeColor[6],
		XtNwedgeColor7, wedgeColor[7],
		XtNwedgeColor8, wedgeColor[8],
		XtNwedgeColor9, wedgeColor[9],
		XtNwedgeColor10, wedgeColor[10],
		XtNwedgeColor11, wedgeColor[11], NULL);
#endif
}
#endif

static void
printStatus(char *msg, int nMoves
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		, int dim, int wedges, int bands
#endif
		)
{
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printState(message, msg);
	intCpy(buff, nMoves);
	printState(movesText, buff);
#else
	char titleDsp[TITLE_LENGTH] = "";
	char dimDsp[3];
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) strncpy(dimDsp, "GL", 3);
	else
		(void) strncpy(dimDsp, "2d", 3);
#else
	dimDsp[0] = '\0';
#endif
#ifdef HAVE_SNPRINTF
	(void) snprintf(titleDsp,
		TITLE_LENGTH, "%s%s.%d: %d@ (%d/%s) - %s",
		progDsp, dimDsp, wedges, bands, nMoves, recordDsp, msg);
#else
	(void) sprintf(titleDsp,
		"%s%s.%d: %d@ (%d/%s) - %s",
		progDsp, dimDsp, wedges, bands, nMoves, recordDsp, msg);
#endif
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
#ifdef HAVE_OPENGL
	XtVaSetValues(XtParent((dim == 2) ? puzzle2d: puzzle3d),
		XtNtitle, titleDsp, NULL);
#else
	XtVaSetValues(XtParent(puzzle2d),
		XtNtitle, titleDsp, NULL);
#endif
#endif
#endif
}

static Boolean
handleSolved(int counter, int wedges, int bands, Boolean orient)
{
	int i = (orient) ? 1 : 0;
	int w = (wedges - MIN_WEDGES) / 2;
	int r = bands - MIN_BANDS;

	if (bands <= MAX_BANDS && (counter < puzzleRecord[i][w][r].score ||
			puzzleRecord[i][w][r].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[i][w][r].score = counter;
		(void) strncpy(puzzleRecord[i][w][r].name, userNameDsp,
			USER_NAME_LENGTH);
		if (orient && (counter < puzzleRecord[!i][w][r].score ||
				puzzleRecord[!i][w][r].score <= NEVER)) {
			puzzleRecord[!i][w][r].score = counter;
			(void) strncpy(puzzleRecord[!i][w][r].name, userNameDsp,
				USER_NAME_LENGTH);
		}
		writeRecords();
		printRecord(wedges, bands, orient, False);
		return True;
	}
	return False;
}

#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
static int
getSpeed(int delay)
{
	return MAX_SPEED + MIN_SPEED - delay - 1;
}

static int
getDelay(int speed)
{
	return MAX_SPEED + MIN_SPEED - speed - 1;
}

static void
orientCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int wedges, bands;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	orient = cbs->set;
#elif defined(HAVE_ATHENA)
	orient = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNwedges, &wedges,
		XtNbands, &bands,
		XtNorient, &old,
		XtNpractice, &practice, NULL);
	if (old == orient)
		return;
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNorient, orient, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNorient, orient, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(wedges, bands, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

static void
practiceCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		int state
#endif
		)
{
	int wedges, bands;
	Boolean orient, practice, old;

#ifdef HAVE_MOTIF
	practice = cbs->set;
#elif defined(HAVE_ATHENA)
	practice = state;
#endif
	XtVaGetValues(puzzle2d,
		XtNwedges, &wedges,
		XtNbands, &bands,
		XtNorient, &orient,
		XtNpractice, &old, NULL);
	if (old == practice)
		return;
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(wedges, bands, orient, practice);
	if (practice)
		messageDsp[0] = '\0';
	else
		(void) strncpy(messageDsp, "Randomize (z) to start",
			MESSAGE_LENGTH);
	printState(message, messageDsp);
}

static void
wedgeCallback(Widget w,
#ifdef HAVE_MOTIF
		int position, XmToggleButtonCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer clientData, XtPointer callData
#endif
		)
{
	int wedges, bands, old;
	Boolean orient, practice;

#ifdef HAVE_MOTIF
	if (!cbs->set)
		return;
#elif defined(HAVE_ATHENA)
	int position = (size_t) clientData;
#endif
	XtVaGetValues(puzzle2d,
		XtNwedges, &old,
		XtNbands, &bands,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	wedges = position * 2 + MIN_WEDGES;
	if (old == wedges)
		return;
#ifdef HAVE_ATHENA
	if (wedgeLabel) {
printf("defaultString [%s]\n", wedgeStrings[position]);
		XtVaSetValues(wedgeLabel,
			XtNlabel, wedgeStrings[position], NULL);
	}
#endif
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNwedges, wedges, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNwedges, wedges, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(wedges, bands, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
bandUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int bands;

	XtVaGetValues(puzzle2d,
		XtNbands, &bands, NULL);
	bands++;
	setScale(bandSlider, bandSliderLabel, bands,
		MIN_BANDS, MAX_BANDS, True);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNbands, bands, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNbands, bands, NULL);
}

static void
bandDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int bands;

	XtVaGetValues(puzzle2d,
		XtNbands, &bands, NULL);
	bands--;
	if (bands < MIN_BANDS) {
		return;
	}
	setScale(bandSlider, bandSliderLabel, bands,
		MIN_BANDS, MAX_BANDS, True);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNbands, bands, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNbands, bands, NULL);
}

#else

static void
bandChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int wedges, bands, old;
	Boolean orient, practice;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	int limit;
	bands = cbs->position;
#else
	bands = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &bands,
			MIN_BANDS, MAX_BANDS, SCROLL_SIZE)) {
		return;
	}
#endif
	XtVaGetValues(puzzle2d,
		XtNwedges, &wedges,
		XtNbands, &old,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	if (old == bands)
		return;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	XtVaGetValues(bandChanger,
		XmNmaximumValue, &limit, NULL);
	if (bands >= limit)
		XtVaSetValues(bandChanger,
			XmNmaximumValue, bands + 1,
			XmNposition, bands, NULL);
#elif defined(HAVE_ATHENA)
	setScale(w, label, bands, MIN_BANDS, MAX_BANDS, False);
#endif
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNbands, bands, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNbands, bands, NULL);
	movesDsp = 0;
	intCpy(buff, movesDsp);
	printState(movesText, buff);
	printRecord(wedges, bands, orient, practice);
	messageDsp[0] = '\0';
	printState(message, messageDsp);
}
#endif

#if defined(HAVE_ATHENA) && defined(USE_SPIN)
static void
speedUpCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed++;
	if (speed > MAX_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

static void
speedDownCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int delay, speed;

	XtVaGetValues(puzzle2d,
		XtNdelay, &delay, NULL);
	speed = getSpeed(delay);
	speed--;
	if (speed < MIN_SPEED) {
		return;
	}
	setScale(speedSlider, speedSliderLabel, speed,
		MIN_SPEED, MAX_SPEED, True);
	delay = getDelay(speed);
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}

#else

static void
speedChangeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XmSpinBoxCallbackStruct *cbs
#else
		XmScaleCallbackStruct *cbs
#endif
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
	int delay, speed, old;

#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	speed = cbs->position;
#else
	speed = cbs->value;
#endif
#elif defined(HAVE_ATHENA)
	Widget label = (Widget) clientData;
	if (!thumbScroll(callData, &speed,
			MIN_SPEED, MAX_SPEED, SCROLL_SIZE)) {
		return;
	}
#endif
	delay = getDelay(speed);
	XtVaGetValues(puzzle2d,
		XtNdelay, &old, NULL);
	if (old == delay)
		return;
#ifdef HAVE_ATHENA
	setScale(w, label, speed, MIN_SPEED, MAX_SPEED, True);
#endif
#ifdef HAVE_OPENGL
	XtVaSetValues(puzzle3d,
		XtNdelay, delay, NULL);
#endif
	XtVaSetValues(puzzle2d,
		XtNdelay, delay, NULL);
}
#endif

static void
puzzlePracticeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(practiceDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
}

static void
puzzleRandomizeCallback(Widget w, XtPointer clientData,
#ifdef HAVE_MOTIF
		XmAnyCallbackStruct *cbs
#elif defined(HAVE_ATHENA)
		XtPointer callData
#endif
		)
{
#ifdef HAVE_MOTIF
	if (cbs->reason == XmCR_OK)
#elif defined(HAVE_ATHENA)
	XtPopdown(randomizeDialog);
#endif
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
}

static void
fileMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_GET;

	if (value == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
}

static void
setSoundCheck(Widget w)
{
	Boolean sound;
	XtVaGetValues(puzzle2d,
		XtNsound, &sound, NULL);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(w, sound, False);
#elif defined(HAVE_ATHENA)
	menuCheck(topLevel, w, sound);
#endif
}

static void
setPerspectiveCheck(Widget w)
{
	Boolean perspective;
	XtVaGetValues(puzzle2d,
		XtNperspective, &perspective, NULL);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(w, perspective, False);
#elif defined(HAVE_ATHENA)
	menuCheck(topLevel, w, perspective);
#endif
}

static void
playMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData) + ACTION_UNDO;

#ifndef EXTRA
	if (value >= ACTION_PRACTICE)
		value += ACTION_RANDOMIZE - ACTION_PRACTICE;
	if (value >= ACTION_INCREMENT)
		value += ACTION_SOUND - ACTION_INCREMENT;
#endif
	XtVaSetValues(puzzle2d,
		XtNmenu, value, NULL);
	if (value == ACTION_SOUND)
		setSoundCheck(w);
	else if (value == ACTION_PERSPECTIVE)
		setPerspectiveCheck(w);
}

static void
helpMenuCallback(Widget w, XtPointer clientData, XtPointer callData)
{
	int value = (int) ((size_t) clientData);
	Widget dialog;

#ifdef HAVE_ATHENA
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
#endif
	switch (value) {
	case 0:
		dialog = descriptionDialog;
		break;
	case 1:
		dialog = featuresDialog;
		break;
	case 2:
		dialog = synopsisDialog;
		break;
	case 3:
		dialog = referencesDialog;
		break;
	case 4:
		dialog = aboutDialog;
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuCallback: %d", value);
			XtWarning(buf);
			free(buf);
			return;
		}
	}
#ifdef HAVE_MOTIF
	XtManageChild(dialog);
#elif defined(HAVE_ATHENA)
	XtPopup(dialog, XtGrabNone);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(dialog), &wmDeleteWindow, 1);
#endif
}
#endif

static void
initialize(
#ifdef WINVER
MballWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int wedges, bands;
	Boolean orient, practice;
	char *userName, *scoreFile;

#ifdef WINVER
	int dim;

	initializePuzzle(w, brush);

	wedges = w->mball.wedges;
	bands = w->mball.bands;
	orient = w->mball.orient;
	practice = w->mball.practice;
	userName = w->mball.userName;
	scoreFile = w->mball.scoreFile;
	dim = w->mball.dim;
	if (strcmp(scoreFile, "")) {
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH - 1);
		scoreFileName[FILE_NAME_LENGTH - 1] = '\0';
	}
#else
	int delay;

	makeEquivalent(&userName, &scoreFile, &wedges, &bands,
		&orient, &practice, &delay);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
	{
		int limit;
		XtVaGetValues(bandChanger,
			XmNmaximumValue, &limit, NULL);
		if (bands >= limit)
			XtVaSetValues(bandChanger,
				XmNposition, bands,
				XmNmaximumValue, bands + 1, NULL);
		else
			XtVaSetValues(bandChanger,
				XmNposition, bands, NULL);
	}
#else
	if (bands > MAX_BANDS)
		XtVaSetValues(bandChanger,
			XmNmaximum, bands, NULL);
	XmScaleSetValue(bandChanger, bands);
#endif
	XmToggleButtonSetState(wedgeChoices[(wedges - MIN_WEDGES) / 2],
		True, False);
	XmToggleButtonSetState(orientToggle, orient, True);
	XmToggleButtonSetState(practiceToggle, practice, True);
	{
		int delay, speed;
		XtVaGetValues(w,
			XtNdelay, &delay, NULL);
		speed = getSpeed(delay);
#ifdef USE_SPIN
		XtVaSetValues(speedChanger,
			XmNposition, speed, NULL);
#else
		XmScaleSetValue(speedChanger, speed);
#endif
	}
	setSoundCheck(soundMenuItem);
	setPerspectiveCheck(perspectiveMenuItem);
#elif defined(HAVE_ATHENA)
	XtVaSetValues(orientToggle,
		XtNstate, orient, NULL);
	XtVaSetValues(practiceToggle,
		XtNstate, practice, NULL);
	setScale(bandSlider, bandSliderLabel, bands,
		MIN_BANDS, MAX_BANDS, False);
	{
		int delay, speed;
		XtVaGetValues(w,
			XtNdelay, &delay, NULL);
		speed = getSpeed(delay);
		setScale(speedSlider, speedSliderLabel, speed,
			MIN_SPEED, MAX_SPEED, True);
	}
	setSoundCheck(soundMenuItem);
	setPerspectiveCheck(perspectiveMenuItem);
#endif
#endif
	SET_STARTED(w, False);
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
	userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH - 1);
		userNameDsp[USER_NAME_LENGTH - 1] = '\0';
#else
		char *login;

#ifdef HAVE_CUSERID
		login = cuserid(NULL);
#else
		login = getlogin();
#endif
		if (login == NULL) {
			userNameDsp[0] = '\0';
		} else {
			(void) strncpy(userNameDsp, login, USER_NAME_LENGTH - 1);
			userNameDsp[USER_NAME_LENGTH - 1] = '\0';
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) strncpy(userNameDsp, "guest", 6);
#endif
	}
	printRecord(wedges, bands, orient, practice);
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, wedges, bands);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, 2, wedges, bands);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, 4, wedges, bands);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzle(MballWidget w, int reason)
#else
static void
puzzleCallback(Widget w, caddr_t clientData,
		mballCallbackStruct *callData)
#endif
{
	int wedgesIndex, wedges, bands;
	Boolean orient, practice, cheat, perspective;
#ifdef WINVER
	int dim = 0;
#else
	Boolean start;
	int reason = callData->reason;
	Widget otherw = (Widget) NULL;
#if defined(HAVE_MOTIF) && defined(USE_SPIN)
	int limit;
#elif defined(HAVE_ATHENA) && defined(USE_POPUP)
	int max;
	char *defaultString;
#endif
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
	int dim = 0, otherdim = 0;
#endif

#ifdef HAVE_OPENGL
	if (w == puzzle2d) {
		otherw = puzzle3d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		dim = 2;
		otherdim = 4;
#endif
	} else if (w == puzzle3d) {
		otherw = puzzle2d;
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA)
		otherdim = 2;
		dim = 4;
#endif
	}
#endif
#endif
	messageDsp[0] = '\0';
#ifdef WINVER
	wedges = w->mball.wedges;
	bands = w->mball.bands;
	orient = w->mball.orient;
	practice = w->mball.practice;
	cheat = w->mball.cheat;
	/*start = w->mball.started;*/
	perspective = w->mball.perspective;
	dim = w->mball.dim;
#else
	XtVaGetValues(w,
		XtNwedges, &wedges,
		XtNbands, &bands,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat,
		XtNperspective, &perspective, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA) && defined(HAVE_OPENGL)
		(void) XIconifyWindow(XtDisplay(shell),
			XtWindow(shell),
			XScreenNumberOfScreen(XtScreen(shell)));
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(practiceDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(practiceDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(randomizeDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(randomizeDialog), &wmDeleteWindow, 1);
#else
		XtVaSetValues(puzzle2d,
			XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#elif defined(HAVE_ATHENA)
		XtPopup(solveDialog, XtGrabNone);
		XSetWMProtocols(XtDisplay(topLevel),
			XtWindow(solveDialog), &wmDeleteWindow, 1);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
		break;
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice",
				MESSAGE_LENGTH);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
			printState(recordText, recordDsp);
#endif
		}
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNdirection, RESTORE_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, RESTORE_DIR, NULL);
#endif
		break;
	case ACTION_CLEAR:
		movesDsp = 0;
		randomized = False;
#ifndef WINVER
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNdirection, CLEAR_DIR, NULL);
		XtVaSetValues(w,
			XtNdirection, CLEAR_DIR, NULL);
#endif
		break;
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move",
			MESSAGE_LENGTH);
		break;
	case ACTION_ILLEGAL:
		if (practice || randomized)
			(void) strncpy(messageDsp, "Illegal move",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp,
				"Randomize (z) to start", MESSAGE_LENGTH);
		break;
#ifndef WINVER
	case ACTION_MOVED:
		if (!callData->control)
			movesDsp++;
		if (otherw != (Widget) NULL) {
#ifdef DEBUG
			(void) printf("MOVED: wedges %d, bands %d, direction %d, control %d\n",
				callData->wedge, callData->band,
				callData->direction, callData->control);
#endif
			XtVaSetValues(otherw,
				XtNwedge, callData->wedge,
				XtNband, callData->band,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast,
				XtNstart, True, NULL);
		}
		SET_STARTED(w, True);
		break;
#endif
	case ACTION_SOLVED:
		if (practice) {
			movesDsp = 0;
		} else if (cheat) {
			(void) strncpy(messageDsp, "No cheating ", 13);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else if (handleSolved(movesDsp, wedges, bands, orient)) {
			(void) strncpy(messageDsp, "Congratulations ", 17);
			(void) strncat(messageDsp, userNameDsp, USER_NAME_LENGTH);
			(void) strncat(messageDsp, "!!", 3);
		} else {
			(void) strncpy(messageDsp, "Solved!", 8);
		}
		SET_STARTED(w, False);
#ifndef WINVER
		if (otherw != (Widget) NULL)
			SET_STARTED(otherw, False);
#endif
		randomized = False;
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
#ifndef WINVER
		if (otherw != (Widget) NULL)
			SET_STARTED(otherw, False);
#endif
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		randomized = False;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize (z) to start",
				MESSAGE_LENGTH);
		printRecord(wedges, bands, orient, practice);
#ifdef WINVER
		w->mball.practice = practice;
		w->mball.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNpractice, practice,
				XtNstart, False, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(practiceToggle, practice, True);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(practiceToggle,
			XtNstate, practice, NULL);
#endif
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
#ifdef WINVER
		w->mball.practice = False;
		w->mball.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNpractice, False,
				XtNstart, False, NULL);
#endif
		break;
#ifdef HAVE_OPENGL
	case ACTION_VIEW:
		{
			int view;

#ifdef WINVER
			view = w->mball.view;
#else
			XtVaGetValues(puzzle3d,
				XtNview, &view, NULL);
#endif
			view = (view + 1) % MAX_VIEWS;
#ifdef WINVER
			w->mball.view = view;
#else
			XtVaSetValues(puzzle3d,
				XtNview, view, NULL);
#endif
		}
		break;
#endif
	case ACTION_INCREMENT:
		movesDsp = 0;
		bands++;
		printRecord(wedges, bands, orient, practice);
#ifdef WINVER
		w->mball.bands = bands;
#else
		XtVaSetValues(w,
			XtNbands, bands, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNbands, bands, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		XtVaGetValues(bandChanger,
			XmNmaximumValue, &limit, NULL);
		if (bands >= limit)
			XtVaSetValues(bandChanger,
				XmNmaximumValue, bands + 1,
				XmNposition, bands, NULL);
		else
			XtVaSetValues(bandChanger,
				XmNposition, bands, NULL);
#else
		if (bands > MAX_BANDS)
			XtVaSetValues(bandChanger,
				XmNmaximum, bands, NULL);
		XmScaleSetValue(bandChanger, bands);
#endif
#elif defined(HAVE_ATHENA)
		setScale(bandSlider, bandSliderLabel, bands,
			MIN_BANDS, MAX_BANDS, False);
#endif
#endif
		break;
	case ACTION_DECREMENT:
		movesDsp = 0;
		bands--;
		printRecord(wedges, bands, orient, practice);
#ifdef WINVER
		w->mball.bands = bands;
#else
		XtVaSetValues(w,
			XtNbands, bands, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNbands, bands, NULL);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
		if (bands > MAX_BANDS)
			XtVaSetValues(bandChanger,
				XmNmaximumValue, bands,
				XmNposition, bands, NULL);
		else
			XtVaSetValues(bandChanger,
				XmNposition, bands, NULL);
#else
		if (bands > MAX_BANDS)
			XtVaSetValues(bandChanger,
				XmNmaximum, bands, NULL);
		XmScaleSetValue(bandChanger, bands);
#endif
#elif defined(HAVE_ATHENA)
		setScale(bandSlider, bandSliderLabel, bands,
			MIN_BANDS, MAX_BANDS, False);
#endif
#endif
		break;
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		printRecord(wedges, bands, orient, practice);
#ifdef WINVER
		w->mball.orient = orient;
#else
		XtVaSetValues(w,
			XtNorient, orient, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(orientToggle, orient, True);
#elif defined(HAVE_ATHENA)
		XtVaSetValues(orientToggle,
			XtNstate, orient, NULL);
#endif
#endif
		break;
	case ACTION_WEDGE2:
	case ACTION_WEDGE4:
	case ACTION_WEDGE6:
	case ACTION_WEDGE8:
	case ACTION_WEDGE10:
	case ACTION_WEDGE12:
		movesDsp = 0;
		wedgesIndex = reason - ACTION_WEDGE2;
		wedges = 2 * wedgesIndex + MIN_WEDGES;
		printRecord(wedges, bands, orient, practice);
#ifdef WINVER
		w->mball.wedges = wedges;
#else
#if defined(HAVE_ATHENA) && !defined(USE_POPUP)
		XtVaSetValues(wedgeChoices[wedgesIndex],
			XtNstate, True, NULL);
#endif
		XtVaSetValues(w,
			XtNwedges, wedges, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNwedges, wedges, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(wedgeChoices[wedgesIndex], True, True);
#elif defined(HAVE_ATHENA) && defined(USE_POPUP)
		max = findMaxLength((char **) wedgeStrings,
			sizeof(wedgeStrings) / sizeof(*wedgeStrings));
		createBlank(&defaultString, max, (char *) wedgeStrings[wedgesIndex], 0);
		XtVaSetValues(wedgeLabel,
			XtNlabel, defaultString, NULL);
		free(defaultString);
#endif
#endif
		break;
#ifndef WINVER
	case ACTION_UNDO:
		movesDsp--;
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNstart, True,
				XtNwedge, callData->wedge,
				XtNband, callData->band,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNstart, True,
				XtNwedge, callData->wedge,
				XtNband, callData->band,
				XtNdirection, callData->direction,
				XtNcontrol, callData->control,
				XtNfast, callData->fast, NULL);
		SET_STARTED(w, True);
		break;
#endif
#if defined(WINVER) && defined(HAVE_OPENGL)
	case ACTION_DIM:
		dim++;
		if (dim == 3)
			dim = 4;
		else if (dim > 4)
			dim = 2;
		w->mball.dim = dim;
		break;
#endif
#ifndef WINVER
	case ACTION_CHEAT:
		XtVaSetValues(w,
			XtNcheat, cheat, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNcheat, cheat, NULL);
		break;
#endif
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	case ACTION_SPEED:
		{
			int oldDelay, delay, speed;

			XtVaGetValues(puzzle2d,
				XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAX_SPEED - MIN_SPEED)
				delay = MAX_SPEED - MIN_SPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			speed = getSpeed(delay);
#ifdef HAVE_MOTIF
#ifdef USE_SPIN
			XtVaSetValues(speedChanger,
				XmNposition, speed, NULL);
#else
			XmScaleSetValue(speedChanger, speed);
#endif
#elif defined(HAVE_ATHENA)
			setScale(speedSlider, speedSliderLabel, speed,
				MIN_SPEED, MAX_SPEED, True);
#endif
		}
		return;
	case ACTION_SOUND:
		setSoundCheck(soundMenuItem);
		break;
#endif
	case ACTION_PERSPECTIVE:
		perspective = !perspective;
#ifdef WINVER
		w->mball.perspective = perspective;
#else
		XtVaSetValues(w,
			XtNperspective, perspective, NULL);
		if (otherw != (Widget) NULL)
			XtVaSetValues(otherw,
				XtNperspective, perspective, NULL);
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
		setPerspectiveCheck(perspectiveMenuItem);
#endif
#endif
		break;
	}
#ifdef WINVER
	printStatus(messageDsp, movesDsp, dim, wedges, bands);
#else
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	printStatus(messageDsp, movesDsp);
#else
	printStatus(messageDsp, movesDsp, dim, wedges, bands);
#ifdef HAVE_OPENGL
	printStatus(messageDsp, movesDsp, otherdim, wedges, bands);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzleMove(MballWidget w, int reason, int wedges, int bands,
		int direction, int control, int fast)
{
	/* Boolean orient, practice, cheat; */
	int dim;

	messageDsp[0] = '\0';
	wedges = w->mball.wedges;
	bands = w->mball.bands;
#if 0
	orient = w->mball.orient;
	practice = w->mball.practice;
	cheat = w->mball.cheat;
#endif
	dim = w->mball.dim;
	switch (reason) {
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_CONTROL:
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	}
	printStatus(messageDsp, movesDsp, dim, wedges, bands);
}

static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.mball.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.mball.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		randomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					TOP,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					BOTTOM,
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			clearPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_PRACTICE:
			practicePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			orientizePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_PERSPECTIVE:
			perspectivePuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
#ifdef HAVE_OPENGL
		case ACTION_DIM:
			(void) dimPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
#endif
		case ACTION_VIEW:
			(void) viewPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_TOP:
		case ACTION_TTR:
		case ACTION_TR:
		case ACTION_RIGHT:
		case ACTION_BR:
		case ACTION_BBR:
		case ACTION_BOTTOM:
		case ACTION_BBL:
		case ACTION_BL:
		case ACTION_LEFT:
		case ACTION_TL:
		case ACTION_TTL:
		case ACTION_CW:
		case ACTION_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_TOP, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_CONTROL_TOP:
		case ACTION_CONTROL_TTR:
		case ACTION_CONTROL_TR:
		case ACTION_CONTROL_RIGHT:
		case ACTION_CONTROL_BR:
		case ACTION_CONTROL_BBR:
		case ACTION_CONTROL_BOTTOM:
		case ACTION_CONTROL_BBL:
		case ACTION_CONTROL_BL:
		case ACTION_CONTROL_LEFT:
		case ACTION_CONTROL_TL:
		case ACTION_CONTROL_TTL:
		case ACTION_CONTROL_CW:
		case ACTION_CONTROL_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) - ACTION_CONTROL_TOP,
					TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_INCREMENT:
			incrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECREMENT:
			if (decrementPuzzle(&widget)) {
				sizePuzzle(&widget);
				(void) InvalidateRect(hWnd, NULL, TRUE);
			}
			break;
		case ACTION_WEDGE2:
		case ACTION_WEDGE4:
		case ACTION_WEDGE6:
		case ACTION_WEDGE8:
		case ACTION_WEDGE10:
		case ACTION_WEDGE12:
			wedgePuzzle(&widget,
				(int) LOWORD(wParam) - ACTION_WEDGE2);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK | MB_ICONQUESTION);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK | MB_ICONEXCLAMATION);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK | MB_ICONINFORMATION);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(XtPointer) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (int) msg.wParam;
}

#else

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*mball.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*mball.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*mball.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*mball.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*mball.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*mball.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*mball.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*mball.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*mball.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*mball.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-wedge0", (char *) "*mball.wedgeColor0", XrmoptionSepArg, NULL},
	{(char *) "-wedge1", (char *) "*mball.wedgeColor1", XrmoptionSepArg, NULL},
	{(char *) "-wedge2", (char *) "*mball.wedgeColor2", XrmoptionSepArg, NULL},
	{(char *) "-wedge3", (char *) "*mball.wedgeColor3", XrmoptionSepArg, NULL},
	{(char *) "-wedge4", (char *) "*mball.wedgeColor4", XrmoptionSepArg, NULL},
	{(char *) "-wedge5", (char *) "*mball.wedgeColor5", XrmoptionSepArg, NULL},
	{(char *) "-wedge6", (char *) "*mball.wedgeColor6", XrmoptionSepArg, NULL},
	{(char *) "-wedge7", (char *) "*mball.wedgeColor7", XrmoptionSepArg, NULL},
	{(char *) "-wedge8", (char *) "*mball.wedgeColor8", XrmoptionSepArg, NULL},
	{(char *) "-wedge9", (char *) "*mball.wedgeColor9", XrmoptionSepArg, NULL},
	{(char *) "-wedge10", (char *) "*mball.wedgeColor10", XrmoptionSepArg, NULL},
	{(char *) "-wedge11", (char *) "*mball.wedgeColor11", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*mball.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*mball.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*mball.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*mball.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*mball.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*mball.font", XrmoptionSepArg, NULL},
	{(char *) "-view", (char *) "*mball.view", XrmoptionSepArg, NULL},
	{(char *) "-perspective", (char *) "*mball.perspective", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noperspective", (char *) "*mball.perspective", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-wedges", (char *) "*mball.wedges", XrmoptionSepArg, NULL},
	{(char *) "-bands", (char *) "*mball.bands", XrmoptionSepArg, NULL},
	{(char *) "-orient", (char *) "*mball.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*mball.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*mball.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*mball.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-base", (char *) "*mball.base", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*mball.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*mball.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*mball.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*mball.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
soundCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	Boolean value = cbs->set;

	XtVaSetValues(puzzle2d,
		XtNsound, value, NULL);
}

static void
perspectiveCallback(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	Boolean value = cbs->set;

	XtVaSetValues(puzzle2d,
		XtNperspective, value, NULL);
}

#elif defined(HAVE_ATHENA)
static void
puzzlePracticeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(practiceDialog);
}

static void
createPracticeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	practiceDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, practiceDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzlePracticeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzlePracticeCancelCallback, dialog);
	XtRealizeWidget(practiceDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(practiceDialog), &wmDeleteWindow, 1);
}

static void
puzzleRandomizeCancelCallback(Widget w, XtPointer clientData,
		XtPointer callData)
{
	XtPopdown(randomizeDialog);
}

static void
createRandomizeQuery(char *title, char *text)
{
	Widget dialog, okDialog, cancelDialog;

	randomizeDialog = XtCreatePopupShell(title,
		transientShellWidgetClass, topLevel, NULL, 0);
	dialog = XtVaCreateManagedWidget("dialog",
		dialogWidgetClass, randomizeDialog,
		XtNlabel, text, NULL);
	okDialog = XtVaCreateManagedWidget("OK",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(okDialog, XtNcallback,
		puzzleRandomizeCallback, dialog);
	cancelDialog = XtVaCreateManagedWidget("Cancel",
		commandWidgetClass, dialog, NULL);
	XtAddCallback(cancelDialog, XtNcallback,
		puzzleRandomizeCancelCallback, dialog);
	XtRealizeWidget(randomizeDialog);
	XSetWMProtocols(XtDisplay(topLevel),
		XtWindow(randomizeDialog), &wmDeleteWindow, 1);
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
	XtAppContext appCon;
#if defined(HAVE_MOTIF) || defined(HAVE_ATHENA)
	Widget mainPanel, controlPanel, menuBar;
	Widget movesRowCol, changerRowCol;
	Widget playMenu;
#endif
#ifdef HAVE_MOTIF
	Widget pullDownMenu, widget;
	Widget menuBarPanel;
	Widget labelRowCol, bandsRowCol, speedChangerRowCol;
	Widget messageRowCol, toggleRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString, clearString;
#ifdef EXTRA
	XmString practiceString;
#endif
	XmString randomizeString, solveString;
#ifdef EXTRA
	XmString incrementString, decrementString, orientString;
	XmString speedString, slowString;
#endif
	XmString soundString, perspectiveString;
#ifdef HAVE_OPENGL
	XmString viewString;
#endif
#elif defined(HAVE_ATHENA)
	XtActionsRec actions[] = {
		{(char *) "DeleteWindowProc", deleteWindowProc},
		{(char *) "ClosePopupPanel", (XtActionProc) closePopupPanel}
	};
	String fallbackResources[] = {
		(char *) "?.translations: #override <Message>WM_PROTOCOLS: DeleteWindowProc()",
		(char *) "?.TransientShell.translations: #override <Message>WM_PROTOCOLS: ClosePopupPanel()",
		NULL
	};
	Widget bandBox, wedgeBox, speedBox;
	Widget orientBox, practiceBox;
	Widget movesLabel, recordLabel;
	Widget wedgesRowCol, toggleRowCol;
	Widget w;
	Widget fileLabel, playLabel, helpLabel;
	int wedges, bands, delay;
	Boolean orient, practice;
	unsigned int i;
#elif defined(HAVE_OPENGL)
	char titleDsp[TITLE_FILE_LENGTH];
#endif

	progDsp = strrchr(argv[0], '/');
	if (progDsp != NULL)
		progDsp++;
	if (progDsp == NULL)
		progDsp = argv[0];
	topLevel = XtVaAppInitialize(NULL, "XMball",
		options, XtNumber(options), &argc, argv,
#ifdef HAVE_ATHENA
		fallbackResources,
#else
		NULL,
#endif
		NULL);
	appCon = XtWidgetToApplicationContext(topLevel);
	if (argc != 1)
		usage(argv[0], synopsisHelp);
#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P', NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "fileMenu",
		0, fileMenuCallback,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL, NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
#ifdef EXTRA
	practiceString = XmStringCreateSimple((char *) "Practice");
#endif
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Auto-solve");
#ifdef EXTRA
	incrementString = XmStringCreateSimple((char *) "Increment Bands");
	decrementString = XmStringCreateSimple((char *) "Decrement Bands");
	orientString = XmStringCreateSimple((char *) "Oriented");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
#endif
	soundString = XmStringCreateSimple((char *) "Sound @");
#ifdef HAVE_OPENGL
	viewString = XmStringCreateSimple((char *) "View");
#endif
	perspectiveString = XmStringCreateSimple((char *) "Perspective");
	playMenu = XmVaCreateSimplePulldownMenu(menuBar, (char *) "playMenu",
		1, playMenuCallback,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
#endif
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
#ifdef EXTRA
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
#endif
		XmVaTOGGLEBUTTON, soundString, '@', NULL, NULL,
#ifdef HAVE_OPENGL
		XmVaPUSHBUTTON, viewString, 'V', NULL, NULL,
#endif
		XmVaTOGGLEBUTTON, perspectiveString, 't', NULL, NULL, NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
#ifdef EXTRA
	XmStringFree(practiceString);
#endif
	XmStringFree(randomizeString);
	XmStringFree(solveString);
#ifdef EXTRA
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientString);
	XmStringFree(speedString);
	XmStringFree(slowString);
#endif
	XmStringFree(soundString);
#ifdef HAVE_OPENGL
	XmStringFree(viewString);
#endif
	XmStringFree(perspectiveString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL); /* mnemonic warning on Cygwin */
	XtVaSetValues(menuBar,
		XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 0);
	widget = XtVaCreateManagedWidget("Features...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 1);
	widget = XtVaCreateManagedWidget("Synopsis...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'S', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 2);
	widget = XtVaCreateManagedWidget("References...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 3);
	widget = XtVaCreateManagedWidget("About...",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuCallback, (char *) 4);
	XtManageChild(menuBar);
	solveDialog = createHelp(menuBar, "Solve", (char *) solveHelp);
	practiceDialog = createQuery(topLevel, "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?",
		(XtCallbackProc) puzzlePracticeCallback);
	randomizeDialog = createQuery(topLevel, "Randomize Query",
		(char *) "Are you sure you want to randomize?",
		(XtCallbackProc) puzzleRandomizeCallback);
	mainPanel = XtVaCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel, NULL);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate and out of date. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNorientation, XmHORIZONTAL, NULL);
	createRadio(labelRowCol, (Widget **) &wedgeChoices[0],
		wedgeStrings, wedgesLabel, XtNumber(wedgeStrings),
		(XtCallbackProc) wedgeCallback);
	bandsRowCol = XtVaCreateManagedWidget("bandsRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		xmRowColumnWidgetClass, bandsRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(changerRowCol, &bandChanger,
		(char *) bandsLabel, DEFAULT_BANDS,
		MIN_BANDS, MAX_BANDS, 2,
		(XtCallbackProc) bandChangeCallback);
#else
	createSlider(changerRowCol, &bandChanger,
		(char *) bandsLabel, DEFAULT_BANDS,
		MIN_BANDS, MAX_BANDS, 2, SCROLL_SIZE,
		(XtCallbackProc) bandChangeCallback);
#endif
	speedChangerRowCol = XtVaCreateManagedWidget("speedChangerRowCol",
		xmRowColumnWidgetClass, bandsRowCol,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
#ifdef USE_SPIN
	createSpinner(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2,
		(XtCallbackProc) speedChangeCallback);
#else
	createSlider(speedChangerRowCol, &speedChanger,
		(char *) speedLabel, MAX_SPEED - DEFAULT_DELAY,
		MIN_SPEED, MAX_SPEED, 2, SCROLL_SIZE,
		(XtCallbackProc) speedChangeCallback);
#endif
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_TIGHT, NULL);
	orientToggle = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_ORIENT, NULL);
	XtAddCallback(orientToggle, XmNvalueChangedCallback,
		(XtCallbackProc) orientCallback, (XtPointer) NULL);
	practiceToggle = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, toggleRowCol,
		XmNset, DEFAULT_PRACTICE, NULL);
	XtAddCallback(practiceToggle, XmNvalueChangedCallback,
		(XtCallbackProc) practiceCallback, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Masterball! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("mball",
		mball2dWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("mball",
		mballGLWidgetClass, mainPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#endif
#elif defined(HAVE_ATHENA)
	XtAppAddActions(appCon, actions, XtNumber(actions));
	createPracticeQuery((char *) "Practice Query",
		(char *) "Are you sure you want to toggle the practice mode?");
	createRandomizeQuery((char *) "Randomize Query",
		(char *) "Are you sure you want to randomize?");
	createHelp(topLevel, &solveDialog, (char *) "Auto-solve",
		(char *) solveHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &descriptionDialog, (char *) "Description",
		(char *) descriptionHelp, (XtCallbackProc) closePopupPanel2);
	createScrollHelp(topLevel, &featuresDialog, (char *) "Features",
		(char *) featuresHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &synopsisDialog, (char *) "Synopsis",
		(char *) synopsisHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &referencesDialog, (char *) "References",
		(char *) referencesHelp, (XtCallbackProc) closePopupPanel2);
	createHelp(topLevel, &aboutDialog, (char *) "About",
		(char *) aboutHelp, (XtCallbackProc) closePopupPanel2);
	mainPanel = XtVaCreateManagedWidget("form",
		panedWidgetClass, topLevel, NULL);
	menuBar = XtVaCreateManagedWidget("MenuBar",
		formWidgetClass, mainPanel,
		XtNborderWidth, 1, NULL);
	createMenu(menuBar, &fileLabel, NULL,
		fileTypes, "File", numFileTypes,
		0, False, fileMenuCallback);
	playLabel = XtVaCreateManagedWidget("Play",
		menuButtonWidgetClass, menuBar,
		XtNborderWidth, 0,
		XtNfromHoriz, fileLabel, NULL);
	playMenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, playLabel, NULL);
	for (i = 0; i < numPlayTypes; i++) {
		if (i == numPlayTypes - 2) {
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu,
				XtNleftMargin, 20, NULL); /* for check */
			soundMenuItem = w;
		} else if (i == numPlayTypes - 1) {
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu,
				XtNleftMargin, 20, NULL); /* for check */
			perspectiveMenuItem = w;
		} else
			w = XtVaCreateManagedWidget(playTypes[i],
				smeBSBObjectClass, playMenu, NULL);
		XtAddCallback(w,
			XtNcallback, (XtCallbackProc) playMenuCallback,
			(XtPointer) (size_t) i);
	}
	createMenu(menuBar, &helpLabel, playLabel,
		helpTypes, "Help", numHelpTypes,
		0, False, helpMenuCallback);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		formWidgetClass, mainPanel,
		XtNborderWidth, 0, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0, NULL);
	movesLabel = XtVaCreateManagedWidget("Moves",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0, NULL);
	movesText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 32,
		XtNfromHoriz, movesLabel, NULL);
	recordLabel = XtVaCreateManagedWidget("Record",
		labelWidgetClass, movesRowCol,
		XtNborderWidth, 0,
		XtNfromHoriz, movesText, NULL);
	recordText = XtVaCreateManagedWidget("0",
		labelWidgetClass, movesRowCol,
		XtNwidth, 108,
		XtNfromHoriz, recordLabel, NULL);
	wedgesRowCol = XtVaCreateManagedWidget("wedgesRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, movesRowCol, NULL);
	wedgeBox = XtVaCreateManagedWidget("wedgeBox",
		boxWidgetClass, wedgesRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	changerRowCol = XtVaCreateManagedWidget("changerRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, wedgesRowCol, NULL);
	bandBox = XtVaCreateManagedWidget("bandBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0, NULL);
	speedBox = XtVaCreateManagedWidget("speedBox",
		boxWidgetClass, changerRowCol,
		XtNorientation, XtorientHorizontal,
		XtNborderWidth, 0,
		XtNfromHoriz, bandBox, NULL);
	toggleRowCol = XtVaCreateManagedWidget("toggleRowCol",
		formWidgetClass, controlPanel,
		XtNborderWidth, 0,
		XtNfromVert, changerRowCol, NULL);
	orientBox = XtVaCreateManagedWidget("Oriented",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNstate, DEFAULT_ORIENT,
		XtNborderWidth, 0, NULL);
	practiceBox = XtVaCreateManagedWidget("Practice",
		boxWidgetClass, toggleRowCol,
		XtNorientation, XtorientHorizontal,
		XtNstate, DEFAULT_PRACTICE,
		XtNborderWidth, 0,
		XtNfromHoriz, orientBox, NULL);
	message = XtVaCreateManagedWidget("Welcome",
		labelWidgetClass, controlPanel,
		XtNjustify,  XtJustifyLeft,
		XtNborderWidth, 0,
		XtNwidth, 180,
		XtNfromVert, toggleRowCol, NULL);
	puzzle2d = XtVaCreateManagedWidget("mball",
		mball2dWidgetClass, mainPanel,
		XtNfromVert, controlPanel,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtVaGetValues(puzzle2d,
		XtNwedges, &wedges,
		XtNbands, &bands,
		XtNorient, &orient,
		XtNpractice, &practice,
		XtNdelay, &delay, NULL);
#ifdef HAVE_OPENGL
	puzzle3d = XtVaCreateManagedWidget("mball",
		mballGLWidgetClass, mainPanel,
		XtNfromVert, puzzle2d,
		XtNheight, WINDOW_HEIGHT, NULL);
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#endif

#ifdef USE_POPUP
	createPopupMenu(wedgeBox, &wedgeLabel, wedgeStrings,
		wedgesLabel, 0, (wedges - MIN_WEDGES) / 2,
		sizeof(wedgeStrings) / sizeof(*wedgeStrings),
		(XtCallbackProc) wedgeCallback);
#else
	createRadio(wedgeBox, (Widget **) &wedgeChoices[0], wedgeStrings,
		wedgesLabel, 0, (wedges - MIN_WEDGES) / 2,
		sizeof(wedgeStrings) / sizeof(*wedgeStrings),
		(XtCallbackProc) wedgeCallback);
#endif
#ifdef USE_SPIN
	createSpinner(bandBox, &bandSliderLabel,
		bandsLabel, 0, bands, MIN_BANDS, MAX_BANDS, False,
		bandUpCallback, bandDownCallback);
	createSpinner(speedBox, &speedSliderLabel,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		speedUpCallback, speedDownCallback);
#else
	createSlider(bandBox, &bandSliderLabel, &bandSlider,
		bandsLabel, 0, bands, MIN_BANDS, MAX_BANDS, False,
		SCROLL_SIZE, bandChangeCallback, bandChangeCallback);
	createSlider(speedBox, &speedSliderLabel, &speedSlider,
		speedLabel, 0, getSpeed(delay), MIN_SPEED, MAX_SPEED, True,
		SCROLL_SIZE, speedChangeCallback, speedChangeCallback);
#endif
	createToggle(orientBox, &orientToggle, "Oriented:", 0,
		orient, (XtCallbackProc) orientCallback);
	createToggle(practiceBox, &practiceToggle, "Practice:", 0,
		practice, (XtCallbackProc) practiceCallback);
	XtRealizeWidget(topLevel);
	if (wmDeleteWindow == None)
		wmDeleteWindow = XInternAtom(XtDisplay(topLevel),
			"WM_DELETE_WINDOW", FALSE);
	XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
		&wmDeleteWindow, 1);
#else
	puzzle2d = XtVaCreateManagedWidget("mball",
		mball2dWidgetClass, topLevel, NULL);
#ifdef HAVE_OPENGL
	shell = XtAppCreateShell(progDsp, titleDsp,
		topLevelShellWidgetClass, XtDisplay(topLevel), NULL, 0);
	puzzle3d = XtVaCreateManagedWidget("mball",
		mballGLWidgetClass, shell, NULL);
	XtAddCallback(puzzle3d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#endif
#endif
	XtVaGetValues(puzzle2d,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel,
			XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle2d, XtNselectCallback,
		(XtCallbackProc) puzzleCallback, (XtPointer) NULL);
#ifdef HAVE_MOTIF
	updateToggle(playMenu, &soundMenuItem,
		False,
#ifdef EXTRA
		11,
#else
		5,
#endif
		(XtCallbackProc) soundCallback);
	updateToggle(playMenu, &perspectiveMenuItem,
		False,
#ifdef HAVE_OPENGL
#ifdef EXTRA
		13,
#else
		7,
#endif
#else
#ifdef EXTRA
		12,
#else
		6,
#endif
#endif
		(XtCallbackProc) perspectiveCallback);
	descriptionDialog = createHelp(menuBar, (char *) "Description",
		(char *) descriptionHelp);
	featuresDialog = createScrollHelp(menuBar, (char *) "Features",
		(char *) featuresHelp, pixmap);
	synopsisDialog = createHelp(menuBar, (char *) "Synopsis",
		(char *) synopsisHelp);
	referencesDialog = createHelp(menuBar, (char *) "References",
		(char *) referencesHelp);
	aboutDialog = createHelp(menuBar, (char *) "About",
		(char *) aboutHelp);
#endif
	initialize(puzzle2d);
	XtRealizeWidget(topLevel);
#if !defined(HAVE_MOTIF) && !defined(HAVE_ATHENA) && defined(HAVE_OPENGL)
	XtRealizeWidget(shell);
#endif
	XGrabButton(XtDisplay(puzzle2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle2d),
		XCreateFontCursor(XtDisplay(puzzle2d), XC_hand2));
	XtAppMainLoop(appCon);

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
