; A lightweight function to read a file's bytes into a stobj array of bytes
;
; Copyright (C) 2021 Kestrel Institute
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(local (include-book "kestrel/bv-lists/unsigned-byte-listp" :dir :system))
;; (include-book "kestrel/utilities/channel-contents" :dir :system)
(local (include-book "file-length-dollar"))
(local (include-book "open-input-channel"))
(local (include-book "kestrel/utilities/channels" :dir :system))
(local (include-book "kestrel/utilities/state" :dir :system))
(local (include-book "read-byte-dollar"))
(local (include-book "kestrel/lists-light/cons" :dir :system))

(local (in-theory (disable assoc-equal
                           channel-contents
                           open-input-channels
                           open-input-channel-p
                           open-input-channel-p1
                           state-p
                           update-nth
                           nth
                           member-equal
                           open-input-channel-any-p1
                           )))

;; A stobj with a single field, which is a byte array.
(defstobj byte-array-stobj
  (bytes :type (array (unsigned-byte 8) (0)) ; initially empty
         :initially 0
         :resizable t))

;; Names generated by defstobj:
(in-theory (disable byte-array-stobjp bytes-length resize-bytes bytesi update-bytesi))

;; Returns (mv byte-array-stobj state).
(defund read-file-into-byte-array-stobj-aux (channel next-index byte-array-stobj state)
  (declare (xargs :guard (and (symbolp channel)
                              (natp next-index)
                              (open-input-channel-p channel :byte state))
                  :stobjs (byte-array-stobj state)
                  :measure (len (cddr (assoc-equal channel (open-input-channels state))) ;;(channel-contents channel state)
                                )
                  :hints (("Goal" :in-theory (enable channel-contents)))
                  :guard-hints (("Goal" :in-theory (enable open-input-channel-p
                                                           )))))
  (if (not (mbt (and (open-input-channel-p channel :byte state) ; for termination
                     (state-p state))))
      (mv byte-array-stobj state)
    (mv-let (maybe-byte state)
      (read-byte$ channel state)
      (if (not maybe-byte) ; no more bytes
          (if (= (bytes-length byte-array-stobj) next-index)
              (mv byte-array-stobj state)
            (prog2$ (er hard? 'read-file-into-byte-array-stobj-aux "Too few bytes in file.") ; should not happen
                    (mv byte-array-stobj state)))
        ;; at least one more byte:
        (if (<= (bytes-length byte-array-stobj) next-index)
            (prog2$ (er hard? 'read-file-into-byte-array-stobj-aux "Too many bytes in file.") ; should not happen
                    (mv byte-array-stobj state))
          (let ((byte-array-stobj (update-bytesi next-index maybe-byte byte-array-stobj)))
            (read-file-into-byte-array-stobj-aux channel
                                            (+ 1 next-index)
                                            byte-array-stobj
                                            state)))))))

(defthm state-p1-of-mv-nth-1-of-read-file-into-byte-array-stobj-aux
  (implies (and (symbolp channel)
                ;; (open-input-channel-p channel :byte state)
                (state-p1 state))
           (state-p1 (mv-nth 1 (read-file-into-byte-array-stobj-aux channel next-index byte-array-stobj state))))
  :hints (("Goal" :in-theory (enable read-file-into-byte-array-stobj-aux
                                     open-input-channel-p
                                     open-input-channel-p1))))

(defthm open-input-channel-p1-of-mv-nth-1-of-read-file-into-byte-array-stobj-aux
  (implies (and (symbolp channel)
                (open-input-channel-p1 channel typ state)
                (state-p1 state))
           (open-input-channel-p1 channel typ (mv-nth 1 (read-file-into-byte-array-stobj-aux channel next-index byte-array-stobj state))))
  :hints (("Goal" :in-theory (enable read-file-into-byte-array-stobj-aux))))

(defthm open-input-channel-any-p1-of-mv-nth-1-of-read-file-into-byte-array-stobj-aux
  (implies (and (symbolp channel)
                (open-input-channel-any-p1 channel state)
                (state-p1 state))
           (open-input-channel-any-p1 channel (mv-nth 1 (read-file-into-byte-array-stobj-aux channel next-index byte-array-stobj state))))
  :hints (("Goal" :in-theory (enable open-input-channel-any-p1))))

;; Returns (mv erp byte-array-stobj state) where either ERP is non-nil (meaning an error
;; occurred) or else the bytes field of BYTE-ARRAY-STOBJ contains the contents of FILENAME.
(defund read-file-into-byte-array-stobj (filename byte-array-stobj state)
  (declare (xargs :guard (stringp filename)
                  :stobjs (byte-array-stobj state)))
  (mv-let (file-length state)
    (file-length$ filename state)
    (if (not file-length)
        (mv `(:failed-to-get-file-length ,filename) byte-array-stobj state)
      (mv-let (channel state)
        (open-input-channel filename :byte state)
        (if (not channel)
            ;; Error:
            (mv `(:could-not-open-channel ,filename) byte-array-stobj state)
          (let ( ;; make the array the right size:
                (byte-array-stobj (resize-bytes file-length byte-array-stobj)))
            (mv-let (byte-array-stobj state)
              (read-file-into-byte-array-stobj-aux channel 0 byte-array-stobj state)
              (let ((state (close-input-channel channel state)))
                (mv nil ; no error
                    byte-array-stobj
                    state)))))))))
